-- Kopiert Änderungen an Inventurdaten in Lagertabellen zurück
CREATE OR REPLACE FUNCTION TLager.lag__inv__finish(
      invnr         varchar,
      mitLagUpdate  boolean
  ) RETURNS void AS $$
  DECLARE
      r                 record;
      SerShould         integer;
      SerCount          integer;
      SerNrLgID         integer;
      lgChanged         boolean;
      lag_rec           record;
      _lg_anz_tot       numeric(12,4);  -- vgl. lag.lg_anztot
      _lifsch_anz       numeric(20,8);  -- vgl. lifsch.l_abgg_uf1
      _wendat_anz       numeric(20,8);  -- vgl. wendat.w_zugang_uf1
      _summe_lag        numeric(12,4);  -- vgl. lag.lg_anztot
      _menge_anzist_uf1 numeric(12,4);  -- vgl. lag.lg_anzist
      _menge_anztot_uf1 numeric(12,4);  -- vgl. lag.lg_anztot
  BEGIN
    -- LagerLog abschalten, da hier explizit selbst geschrieben wird.
    PERFORM TSystem.Settings__Set('lagerlog_' || current_user, 'F');

    -- Unvollständige Inventur abfangen
    IF      mitLagUpdate
        AND EXISTS (SELECT true FROM invlag WHERE il_iv_nr = invnr AND il_zaedat IS NULL)
    THEN
        -- Zum Abschluss einer Inventur müssen die Zähldaten zu allen Artikeln gepflegt sein.
        RAISE EXCEPTION '%', lang_text(32609);
    END IF;

    -- konkurrierende Inventur(en) vorhanden?
    IF TLager.lag__inv__konkurrierende_inventur__exists( invnr ) THEN -- ivnr version
        -- Inventur kann nicht abgeschlossen werden, da es Konflikte mit anderen offenen Inventuren gibt.
        -- Bitte beachten Sie die farbige Markierung des betroffenen Artikels.
        RAISE EXCEPTION '%', lang_text(32637);
    END IF;

    -- Alle Positionen der Inventur durchgehen
    FOR r IN
        SELECT
          iv_bdat + iv_bdat_time AS iv_start,
          invlag.*,
          art.ak_sernrreq
        FROM inv
          JOIN invlag ON iv_nr = il_iv_nr
          JOIN art ON il_aknr = ak_nr
        WHERE il_iv_nr = invnr
    LOOP
        -- Fehler, wenn keine ME angg. ist.
        IF r.il_mce IS NULL THEN
            -- FinishInventur: Artikel %, % hat keine gültige Mengeneinheit
            RAISE EXCEPTION '%', format( lang_text( 29176 ), r.il_aknr, GetArtBez( r.il_aknr ) );
        END IF;

        -- erst hier nach Prüfung il_mce is null
        _menge_anzist_uf1 := tartikel.me__menge__in__menge_uf1( r.il_mce, r.il_anzist );
        _menge_anztot_uf1 := tartikel.me__menge__in__menge_uf1( r.il_mce, r.il_anztot );

        SerNrLgID := null;

        lgChanged :=
            EXISTS(
                SELECT true
                FROM lag
                  JOIN invlag ON lg_id = il_lg_id
                WHERE il_id = r.il_id
                  AND (
                        lg_ort  <> il_ort
                    OR  lg_chnr <> il_chnr
                    OR  lg_aknr <> il_aknr
                  )
            )
        ;

        -- Fall 1: Lagerort, Charge oder Artikelnummer geändert.
        -- => Löschen Ausgangslagerort und neuen anlegen mit Inventurmenge.
        IF lgChanged AND NOT r.il_delete THEN

            -- Neu anlegen mit Inventurmenge, lg_id merken
            INSERT INTO lag (
                lg_aknr,    lg_anztot,          lg_anzist,          lg_mce,
                lg_ort, lg_chnr,    lg_dim1,    lg_dim2,      lg_dim3,
                lg_zaedat,    lg_konsDat,   lg_konsEnde,
                lg_sperr,   lg_regalsta,    lg_txt
              )
            VALUES (
                r.il_aknr,  _menge_anzist_uf1,  _menge_anzist_uf1,  r.il_mce,
                r.il_ort, r.il_chnr,  r.il_dim1,  r.il_dim2,  r.il_dim3,
                r.il_zaedat,  r.il_konsDat, r.il_konsEnde,
                r.il_sperr, r.il_regalsta,  r.il_txt
              )
            RETURNING lg_id INTO SerNrLgID;

            -- Seriennummern umhängen
            UPDATE lagsernr SET
              lgs_lg_id = SerNrLgID
            WHERE lgs_lg_id = r.il_lg_id;

            -- Alten Lagerort merken (für LagerLog) und löschen.
            lag_rec := null;

            SELECT
              lg_aknr, lg_ort, lg_chnr
            FROM lag
            WHERE lg_id = r.il_lg_id
            INTO lag_rec;

            -- alten Lager-Eintrag löschen
            DELETE FROM lag
            WHERE lg_id = r.il_lg_id;

            -- Log-Eintrag Ziel-Artikel
            -- Durch '?=I' markieren, dass es hier eine Inventuränderung gab und umgebucht, oder Artikel geändert wurde.
            INSERT INTO lagerlog (
                lo_aknr_old,      lo_aknr_new,  lo_aktion,  lo_user,                lo_txt,
                lo_stk_old,         lo_stk_new,         lo_lg_bestand,      lo_lgort_old,
                lo_lgort_new, lo_lgchnr_old,    lo_lgchnr_new
              )
            VALUES (
                lag_rec.lg_aknr,  r.il_aknr,    '?=I',      'INV ' || current_user, r.il_iv_nr,
                _menge_anztot_uf1,  _menge_anzist_uf1,  _menge_anzist_uf1,  lag_rec.lg_ort,
                r.il_ort,     lag_rec.lg_chnr,  r.il_chnr
              )
            ;

            -- Wenn Artikel geändert wird
            IF lag_rec.lg_aknr <> r.il_aknr THEN
                -- Eintrag Quell-Artikel
                -- '--I' markiert den LO als gelöscht durch Inventur
                INSERT INTO lagerlog (
                    lo_aknr_new,      lo_aktion,  lo_user,                lo_txt,
                    lo_stk_old,         lo_stk_new, lo_lg_bestand,
                    lo_lgort_new,   lo_lgchnr_new
                  )
                VALUES (
                    lag_rec.lg_aknr,  '--I',      'INV ' || current_user, r.il_iv_nr,
                    _menge_anztot_uf1,  0,          0,
                    lag_rec.lg_ort, lag_rec.lg_chnr
                  )
                ;

            END IF;

            -- Debugging
            -- RAISE WARNING 'Fall 1 - Lag-Change: % , % , % ', r.il_aknr, r.il_ort, r.il_chnr;
        END IF;


        -- Fall 2: Bei Inventur neu angelegt
        -- => Neu erstellen in Lag
        IF
                NOT r.il_delete
            AND NOT EXISTS(
                  SELECT true
                  FROM lag
                  WHERE lg_aknr = r.il_aknr
                    AND lg_chnr = r.il_chnr
                    AND lg_ort  = r.il_ort
                    AND lg_dim1 = r.il_dim1
                    AND lg_dim2 = r.il_dim2
                    AND lg_dim3 = r.il_dim3
                )
        THEN

            -- Neu anlegen mit Inventurmenge, lg_id merken
            INSERT INTO lag (
                lg_aknr,    lg_anztot,          lg_anzist,          lg_mce,
                lg_ort,   lg_chnr,    lg_dim1,    lg_dim2,    lg_dim3,
                lg_zaedat,    lg_konsDat,   lg_konsEnde,
                lg_sperr,   lg_regalsta,    lg_txt
              )
            VALUES (
                r.il_aknr,  _menge_anzist_uf1,  _menge_anzist_uf1,  r.il_mce,
                r.il_ort, r.il_chnr,  r.il_dim1,  r.il_dim2,  r.il_dim3,
                r.il_zaedat,  r.il_konsDat, r.il_konsEnde,
                r.il_sperr, r.il_regalsta,  r.il_txt
              )
            RETURNING lg_id INTO SerNrLgID;

            -- Log-Eintrag
            -- '++I' markiert den als neu angelegt durch Inventur
            INSERT INTO lagerlog (
                lo_aknr_new,  lo_aktion,  lo_user,                lo_txt,
                lo_stk_old,         lo_stk_new,         lo_lg_bestand,
                lo_lgort_new, lo_lgchnr_new
              )
            VALUES (
                r.il_aknr,    '++I',      'INV ' || current_user, r.il_iv_nr,
                _menge_anztot_uf1,  _menge_anzist_uf1,  _menge_anzist_uf1,
                r.il_ort,     r.il_chnr
              )
            ;

            -- Debug
            -- RAISE WARNING 'Fall 2 - Lag-Insert:  % , % , % ', r.il_aknr, r.il_ort, r.il_chnr;

        -- Fall 3 und 4
        ELSE
            -- Fall 3: Löschflag für Lagerplatz gesetzt
            IF r.il_delete THEN
                -- IF NOT r.ak_sernrreq THEN
                    -- Seriennummern wegwerfen
                    DELETE FROM lagsernr
                     WHERE lgs_lg_id
                        IN (SELECT lg_id FROM lag
                             WHERE lg_aknr = r.il_aknr
                              AND lg_ort  = r.il_ort
                              AND lg_chnr = r.il_chnr
                              AND lg_dim1 = r.il_dim1
                              AND lg_dim2 = r.il_dim2
                              AND lg_dim3 = r.il_dim3
                             )
                    ;
                    -- Löschen aus Lag
                    DELETE FROM lag
                    WHERE lg_aknr = r.il_aknr
                      AND lg_ort  = r.il_ort
                      AND lg_chnr = r.il_chnr
                      AND lg_dim1 = r.il_dim1
                      AND lg_dim2 = r.il_dim2
                      AND lg_dim3 = r.il_dim3
                    ;

                    -- Log-Eintrag
                    -- '--I' markiert den als gelöscht durch Inventur
                    INSERT INTO lagerlog (
                        lo_aknr_new,  lo_aktion,  lo_user,                lo_txt,
                        lo_stk_old,         lo_stk_new, lo_lg_bestand,
                        lo_lgort_new, lo_lgchnr_new
                      )
                    VALUES (
                        r.il_aknr,    '--I',      'INV ' || current_user, r.il_iv_nr,
                        _menge_anztot_uf1,  0,          0,
                        r.il_ort,     r.il_chnr
                      )
                    ;

                    -- Debug
                    -- RAISE WARNING 'Fall 3 - Lag-Delete:  % , % , % ', r.il_aknr, r.il_ort, r.il_chnr;
                -- END IF;

            -- Fall 4: Update eines bestehenden Lagerortes
            ELSE
                IF
                        NOT lgChanged
                    AND r.il_lg_id IS NOT NULL
                THEN
                    -- Update, aber nur bei Umschreiben Stückzahl und wenn sich Lagerort, Charge und Artikel nicht geändert haben
                    UPDATE lag SET
                      lg_anztot = _menge_anzist_uf1,
                      lg_anzist = _menge_anzist_uf1,
                      lg_ort  = r.il_ort,
                      lg_chnr = r.il_chnr,
                      lg_dim1 = r.il_dim1, lg_dim2 = r.il_dim2, lg_dim3 = r.il_dim3,
                      lg_konsdat = r.il_konsDat, lg_konsEnde = r.il_konsEnde
                    WHERE lg_aknr = r.il_aknr
                      AND lg_ort  = r.il_ort
                      AND lg_chnr = r.il_chnr
                      AND lg_dim1 = r.il_dim1
                      AND lg_dim2 = r.il_dim2
                      AND lg_dim3 = r.il_dim3 -- warum nicht per ID r.il_lg_id?
                    ;

                    SerNrLgID := r.il_lg_id;
                    lag_rec := null;

                    SELECT
                      lg_aknr, lg_ort, lg_chnr
                    FROM lag
                    WHERE lg_id = r.il_lg_id
                    INTO lag_rec;

                    -- Log-Eintrag
                    -- Lagerlogeintrag mit '==I' => Reine Bestandsänderung durch Inventur auf bestehendem Lagerort
                    INSERT INTO lagerlog (
                        lo_aknr_old,      lo_aknr_new,  lo_aktion,  lo_user,                lo_txt,
                        lo_stk_old,         lo_stk_new,         lo_lg_bestand,
                        lo_lgort_old,   lo_lgort_new, lo_lgchnr_old,    lo_lgchnr_new
                      )
                    VALUES (
                        lag_rec.lg_aknr,  r.il_aknr,    '==I',      'INV ' || current_user, r.il_iv_nr,
                        _menge_anztot_uf1,  _menge_anzist_uf1,  _menge_anzist_uf1,
                        lag_rec.lg_ort, r.il_ort,     lag_rec.lg_chnr,  r.il_chnr
                      )
                    ;

                    -- Debug
                    -- RAISE WARNING 'Fall 4 - STK-Change:  % , % , % ', r.il_aknr, r.il_ort, r.il_chnr;
                END IF;

                -- Die Differenzen aus zwischenzeitlichen Lagerbewegungen nachziehen, siehe #12619
                IF mitLagUpdate THEN
                    _lg_anz_tot :=
                          lg_anztot
                        FROM lag
                        WHERE lg_aknr = r.il_aknr
                          AND lg_ort  = r.il_ort
                          AND lg_chnr = r.il_chnr
                          AND lg_dim1 = r.il_dim1
                          AND lg_dim2 = r.il_dim2
                          AND lg_dim3 = r.il_dim3
                    ;

                    _wendat_anz :=
                          sum( w_zugang_uf1 )
                        FROM wendat
                        WHERE w_aknr    = r.il_aknr
                          AND w_lgort   = r.il_ort
                          AND w_lgchnr  = r.il_chnr
                          -- Zugang nach Inventur-Start
                          AND w_zug_dat >= r.iv_start
                    ;

                    _lifsch_anz :=
                          sum( l_abgg_uf1 )
                        FROM lifsch
                        WHERE l_aknr    = r.il_aknr
                          AND l_lgort   = r.il_ort
                          AND l_lgchnr  = r.il_chnr
                          -- Abgang nach Inventur-Start
                          AND l_ldat    >= r.iv_start
                    ;

                    _lg_anz_tot := coalesce( _lg_anz_tot, 0 );
                    _wendat_anz := coalesce( _wendat_anz, 0 );
                    _lifsch_anz := coalesce( _lifsch_anz, 0 );

                    -- gesamte Beständsänderung durch Lagerbewegungen
                    -- nie unter 0, wird aber in Oberfläche angezeigt
                    _summe_lag := greatest( (_lg_anz_tot + _wendat_anz - _lifsch_anz), 0 );

                    -- Lagerbewegungen vorhanden
                    IF (_wendat_anz <> 0 OR _lifsch_anz <> 0) THEN
                        -- Aktualisierung des Lagerbestands entspr. gesamter Lagerbewegungen
                        INSERT INTO lag (
                            lg_aknr,   lg_ort,   lg_anztot,   lg_mce,   lg_chnr,    lg_dim1,    lg_dim2,    lg_dim3
                          )
                        VALUES (
                            r.il_aknr, r.il_ort, _summe_lag,  r.il_mce, r.il_chnr,  r.il_dim1,  r.il_dim2,  r.il_dim3
                          )
                        ON CONFLICT( lg_aknr, lg_ort, lg_chnr, lg_dim1, lg_dim2, lg_dim3 ) -- siehe UNIQUE-Index
                        DO UPDATE SET lg_anztot = excluded.lg_anztot
                        ;

                        -- #13799 Korrektur Lagerlog
                        UPDATE lagerlog SET
                          -- Lagerbestand nach den Lagerbewegungen
                          lo_stk_new    = _summe_lag,
                          lo_lg_bestand = _summe_lag,
                          -- Artikelbestand macht lagerlog__b_iu automatisch
                          lo_bestand = NULL
                        WHERE lo_id = (
                                -- letzter Lagerlog-Eintrag der Inventur
                                SELECT lo_id
                                FROM lagerlog
                                WHERE lo_aknr_new = r.il_aknr
                                  AND lo_aktion LIKE '%I' -- ++I, --I, ==I
                                  AND lo_datum = now()::timestamp(0)
                                ORDER BY lo_datum DESC, lo_id DESC
                                LIMIT 1
                              )
                        ;

                    END IF;

                END IF;

            END IF;

        END IF;

        /* Unklar ob das reinkann. Solange auskommentiert. -- Lagerort/Charge/Artikel geändert, oder neuer Lagerort, oder Menge erhöht -> Fehlende Seriennummern generieren.
        IF SerNrLgID IS NOT NULL AND r.ak_sernrreq THEN
          SerCount  := COUNT(1)+1 FROM lagsernr WHERE lgs_lg_id = SerNrLgID;
          SerCount  := COALESCE(NUMSER,1); --Anzahl vorhandene Seriennummern
          SerShould := CEILING(TARTIKEL.ME__MENGE__IN__MENGE_UF1(r.il_mce,r.il_anzist))::INTEGER;
          --pro stück ein Seriennummerndatensatz vorbereiten
          FOR I IN SerCount..SerShould LOOP
            INSERT INTO lagsernr (lgs_lg_id, lgs_sernr) VALUES (SerNrLgID, '~INVENTUR~'||r.il_ak_nr::VARCHAR||' #'||lpad(I::VARCHAR,3,'0'));
          END LOOP;
        END IF;
        */

    END LOOP;

    UPDATE inv SET
      iv_edat = current_date,
      iv_edat_time = current_time,
      iv_def = true
    WHERE iv_nr = invnr
    ;

    PERFORM TSystem.Settings__Set('lagerlog_' || current_user, 'T');

    RETURN;
  END $$ LANGUAGE plpgsql;
--

--
CREATE OR REPLACE FUNCTION Z_99_Deprecated.finishinventur(invnr VARCHAR, mitLagUpdate BOOLEAN) RETURNS VOID AS $$
  BEGIN
    SELECT TLager.lag__inv__finish(invnr, mitLagUpdate);
  END $$ LANGUAGE plpgsql;
--

-- Funktion zur Ermittlung der tatsächlichen Lagermenge: gezählte Menge in der Inventur + Lagerzugänge nach Zählzeitpunkt - Lagerabgänge nach Zählzeitpunkt
CREATE OR REPLACE FUNCTION TLager.lag__inv__get_lagerbewegungen(invlag_rec invlag) RETURNS NUMERIC(12,2) as $$
  DECLARE
    _inv_start TIMESTAMP WITHOUT TIME ZONE;
    _inv_ende  TIMESTAMP WITHOUT TIME ZONE;
    _anz_wendat NUMERIC(12,2);
    _anz_lifsch NUMERIC(12,2);
  BEGIN
    -- Startzeitpunkt der Inventur, dient auch als Beginnzeitpunkt der Lagerbewegungen nach Eintragung des Zähldatums
    -- Zwischen Start und Eintragung des Zähldatums ist das Lager für eventuelle Lagerbewegungen gesperrt
    SELECT iv_bdat + iv_bdat_time, iv_edat + iv_edat_time
    INTO _inv_start, _inv_ende
    FROM inv
    WHERE iv_nr = invlag_rec.il_iv_nr;

    -- Alle Lagerzugänge
    SELECT COALESCE(SUM(w_zugang_uf1), 0)
    INTO _anz_wendat
    FROM wendat
    WHERE w_aknr = invlag_rec.il_aknr
      AND w_lgort = invlag_rec.il_ort
      AND w_lgchnr = invlag_rec.il_chnr
      AND w_zug_dat >= _inv_start
      AND (_inv_ende IS NULL OR w_zug_dat < _inv_ende);

    -- Alle Lagerabgänge
    SELECT COALESCE (SUM(l_abgg_uf1), 0)
    INTO _anz_lifsch
    FROM lifsch
    WHERE l_aknr = invlag_rec.il_aknr
      AND l_lgort = invlag_rec.il_ort
      AND l_lgchnr = invlag_rec.il_chnr
      AND l_ldat >= _inv_start
      AND (_inv_ende IS NULL OR l_ldat < _inv_ende);

    RETURN invlag_rec.il_anzist + _anz_wendat - _anz_lifsch;
  END $$ LANGUAGE plpgsql;
--

--DOKU
--http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Inventur
--DOKU

-- Kopiert aktuelle Lagerdaten in inventurtabelle
CREATE OR REPLACE FUNCTION TLager.lag__inv__start(invnr VARCHAR, whereStr VARCHAR, emptyIst BOOLEAN DEFAULT FALSE) RETURNS INTEGER AS $$
  BEGIN
    -- Kopfdaten für Inventur
    INSERT INTO inv (iv_nr, iv_bdat) VALUES (invnr, current_timestamp);

    -- kopieren Lagerdaten in Inventurbestand ===> Where String = Einschränkung Inventurumfang aus RTF1003
    EXECUTE concat(E'INSERT INTO invlag (il_iv_nr, il_aknr, il_lg_id, il_ort, il_anztot, il_anzist, il_mce, il_konsDat, il_konsEnde, il_chnr, il_dim1, il_dim2, il_dim3, il_sperr, il_regalsta, il_ak_vkpbas, il_ak_hest, il_SerNrString)
      SELECT $1, lg_aknr, lg_id, COALESCE(lg_ort,''''), COALESCE(lg_anztot,0), COALESCE(IfThen($2,0,lg_anztot),0),
        tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(ak_nr), lg_konsDat, lg_konsEnde, COALESCE(lg_chnr,''''), COALESCE(lg_dim1,0), COALESCE(lg_dim2,0), COALESCE(lg_dim3,0),
        COALESCE(lg_sperr,false), COALESCE(lg_regalsta,0), ak_vkpbas, ak_hest, sernr.sernrstring
      FROM lag
      JOIN art    ON ak_nr = lg_aknr
      JOIN artcod ON ak_ac = ac_n
      LEFT JOIN LATERAL (SELECT string_agg(lgs_sernr, ''\n'') AS sernrstring FROM lagsernr WHERE lgs_lg_id = lag.lg_id) AS sernr ON ak_sernrreq
      ', whereStr) USING invnr, emptyIst;

    RETURN (SELECT count(*) FROM invlag WHERE il_iv_nr=invnr);
  END $$ LANGUAGE plpgsql;

  CREATE OR REPLACE FUNCTION Z_99_Deprecated.StartInventur(invnr VARCHAR, whereStr VARCHAR, emptyIst BOOLEAN DEFAULT FALSE) RETURNS INTEGER AS $$
  BEGIN
    RETURN TLager.lag__inv__start(invnr, whereStr, emptyIst);
  END $$ LANGUAGE plpgsql;
--

--
CREATE OR REPLACE FUNCTION TLager.lag__inv__konkurrierende_inventur__exists(
      _aknr                                     varchar,
      _ort                                      varchar,
      _chnr                                     varchar,
      _dim1                                     numeric,
      _dim2                                     numeric,
      _dim3                                     numeric,
      -- Diese nicht aber alle anderen Inventuren berücksichtigen (konkurrierende Inventuren). Default alle.
      _iv_nr__excluded                          varchar = null,
      -- Option: zusätzliche Prüfung, ob Zähldatum der Position offen ist.
      _il_zaedat__offen__check                  boolean = false,

      OUT inv__konkurrierende_inventur__exists  boolean,  -- es gibt konkurrierende Inventur(en)
      OUT inv__konkurrierende_inventur__list    varchar[] -- Auflistung der konkurrierenden Inventur(en)
  ) RETURNS record AS $$
    -- Ermittelt Existenz und Liste offener, konkurrierender Inventuren, siehe #13802.
      -- Über alle Inventuren (_ivnr is null). Fall:  Lagereintrag, welcher in offener Inventur ist.
      -- Oder über alle anderen Inventuren außer der anggebenen Inventur. Fall: konkurrierende Inventuren.
      -- Zusätzliche Prüfung möglich, ob das Zähldatum offen ist. Fall: Änderung am Lagerort darf ggf. erfolgen.

    SELECT
      -- inv__konkurrierende_inventur__exists
      count(*) > 0,
      -- inv__konkurrierende_inventur__list
      array_agg( DISTINCT inv.iv_nr ORDER BY inv.iv_nr )
    FROM invlag
      JOIN inv ON iv_nr = il_iv_nr
    WHERE il_aknr = _aknr
      -- _ort und _chnr müssen zu einem Leerstring normalisiert werden, da Spalten als NOT NULL DEFAULT '' angelegt sind
      AND il_ort  = coalesce( _ort, '' )
      AND il_chnr = coalesce( _chnr, '' )
      AND il_dim1 = _dim1
      AND il_dim2 = _dim2
      AND il_dim3 = _dim3

      -- alle anderen Inventuren oder alle
      AND inv.iv_nr IS DISTINCT FROM _iv_nr__excluded

      -- Inventur ist ohne Abschluss-Datum, also offen
      AND iv_edat IS NULL

      -- Option: zusätzliche Prüfung, ob Zähldatum der Position offen ist.
      AND  (
              _il_zaedat__offen__check IS NOT TRUE  -- null wird wie false (Default) behandelt.
          OR  il_zaedat IS NULL
      )
    ;


  $$ LANGUAGE sql;
--

--
CREATE OR REPLACE FUNCTION TLager.lag__inv__konkurrierende_inventur__exists(
      _ivnr   varchar
  ) RETURNS boolean AS $$
    -- Ermittelt zu den eigenen Inventur-Pos. die Existenz offener, konkurrierender Inventuren, siehe #13802.
      -- Ausgangspunkt: eigene Inventur-Nr.
      -- Es werden alle eigenen Inventur-Pos. auf andere und offene (konkurrierende) Inventur-Pos. geprüft.


    SELECT
      EXISTS(
          SELECT true
          FROM invlag
            JOIN LATERAL
                TLager.lag__inv__konkurrierende_inventur__exists(
                    il_aknr,
                    il_ort,
                    il_chnr,
                    il_dim1,
                    il_dim2,
                    il_dim3,
                    il_iv_nr  -- Alle anderen Inventuren prüfen.
                ) ON true
          WHERE il_iv_nr = _ivnr
            AND inv__konkurrierende_inventur__exists
      )
    ;


  $$ LANGUAGE sql;
--

--
CREATE OR REPLACE FUNCTION TLager.lag__inv__konkurrierende_inventur__exists(
      _lag                                      lag,
      -- Option: zusätzliche Prüfung, ob Zähldatum offen ist.
      _il_zaedat__offen__check                  boolean = false,

      OUT inv__konkurrierende_inventur__exists  boolean,  -- es gibt konkurrierende Inventur(en)
      OUT inv__konkurrierende_inventur__list    varchar[] -- Auflistung der konkurrierenden Inventur(en)
  ) RETURNS record AS $$
    -- Ermittelt Existenz und Liste offener, konkurrierender Inventuren zum Lagereintrag, siehe #13802.
      -- Zusätzliche Prüfung möglich, ob das Zähldatum offen ist. Fall: Änderung am Lagerort darf ggf. erfolgen.


    SELECT
      result.inv__konkurrierende_inventur__exists,
      result.inv__konkurrierende_inventur__list
    FROM
      TLager.lag__inv__konkurrierende_inventur__exists(
          _lag.lg_aknr,
          _lag.lg_ort,
          _lag.lg_chnr,
          _lag.lg_dim1,
          _lag.lg_dim2,
          _lag.lg_dim3,
          null,
          -- Option: zusätzliche Prüfung, ob Zähldatum offen ist.
          _il_zaedat__offen__check
      ) AS result
    ;


  $$ LANGUAGE sql;
---
--- #20834 Anbindung Storetec-Werkzeugschrank
CREATE OR REPLACE FUNCTION tlager.lag__inv__lagartikelkonf__werkzeugschrank_reset(
        IN _lgo_name        varchar, -- Lagerort
        IN _modified_by     varchar  -- User
    ) RETURNS void AS $$
 BEGIN

    --PERFORM disablemodified();
    --PERFORM disablebedarfberech(); --> wird von bestandsabgleich_intern ignoriert. Ansonsten müßte über prepare_artikel_bedarf und do_artikel_bedarf gelaufen werden

    -- Alle Werkzeuge Terminal 1 Mindestanzahl auf 0 setzen, um beim einlesen korrekte Bestände zu erhalten -> wegen Bestellvorschlägen
    UPDATE lagartikelkonf SET lgoa_melde = 0, lgoa_soll = 0, modified_by = _modified_by, modified_date = current_date
        WHERE lgoa_lgo_name LIKE _lgo_name || '%'
          AND ( (lgoa_melde <> 0) OR (lgoa_soll <> 0) )-- Hint: => wird im Trigger bereits berücksichtigt und dann nicht aktualisiert
    ;

    UPDATE lag SET lg_anztot = 0, modified_by = _modified_by, modified_date = current_date
        WHERE lg_ort LIKE _lgo_name || '%'
          AND lg_anztot <> 0
    ;

    --PERFORM enablebedarfberech(); -- siehe oben
    --PERFORM enablemodified();

 END $$ LANGUAGE plpgsql VOLATILE;

---
CREATE OR REPLACE FUNCTION tlager.lag__inv__lagartikelkonf__werkzeugschrank_import(
        IN _lgo_name        varchar,  -- Lagerort
        IN _lgo_desc        varchar,  -- Lagerortbezeichnung
        IN _modified_by     varchar,  -- User
        IN _ak_ac           varchar,  -- AC
        IN _terminal        varchar,  -- ab hier, Felder des Imports
        IN _schrank         varchar,
        IN _fach            varchar,
        IN _sektion         varchar,
        IN _matnr           varchar,
        IN _matbez          varchar,
        IN _beschr          varchar,
        IN _altmat          varchar,
        IN _hest            varchar,
        IN _vpe             varchar,
        IN _preisanz        varchar,
        IN _preisneu        varchar,
        IN _preisgebr       varchar,
        IN _liefkenn        varchar,
        IN _firma           varchar,
        IN _krzbez          varchar,
        IN _bestnr          varchar,
        IN _gebrstufe       varchar,
        IN _kapaz           varchar,
        IN _bestand         varchar,
        IN _bestandmelde    varchar,
        IN _bestandsoll     varchar,
        IN _provpe          varchar
    ) RETURNS void AS $$
 BEGIN

    --- Addition der Bestände aller Artikel einer Datei, funktioniert nicht mit 2 verschiedenen Werkzeugschrank-Dateien mit denselben Artikeln
    INSERT INTO art (ak_ac, ak_nr, ak_bez, ak_standard_mgc, ak_los, insert_by, insert_date)
        SELECT
          _ak_ac,
          TRIM(_matnr),
          TRIM(trailing ',' from TRIM(TRIM(_matbez) || ', ' || TRIM(_beschr))),
          1,
          CAST(_provpe AS NUMERIC(12,4)),
          _modified_by,
          current_date
        WHERE
            NOT EXISTS (SELECT 1 FROM art WHERE ak_nr = TRIM(_matnr))
    ;

    INSERT INTO lagerorte (lgo_name, lgo_desc)
        SELECT
          _lgo_name,
          _lgo_desc
        WHERE NOT EXISTS (SELECT 1 FROM lagerorte WHERE lgo_name = _lgo_name)
    ;

    UPDATE lag SET lg_anztot = lg_anztot + CAST(_bestand AS NUMERIC(12,4)),
                   modified_by=_modified_by,
                   modified_date = current_date
        WHERE lg_aknr = TRIM(_matnr)
          AND lg_ort = _lgo_name
          AND lg_chnr  = TRIM(_gebrstufe)
    ;

    INSERT INTO lag (lg_aknr, lg_ort, lg_mce, lg_chnr, lg_anztot, insert_by, insert_Date)
        SELECT
          TRIM(_matnr),
          _lgo_name,
          standard_mgc_id(TRIM(_matnr)),
          TRIM(_gebrstufe),
          CAST(_bestand AS NUMERIC(12,4)),
          _modified_by,
          current_date
        WHERE     -- Optimierung, nur neu einfügen wenn noch nicht vorhanden
            NOT EXISTS (SELECT 1 FROM lag
                WHERE lg_aknr = TRIM(_matnr)
                AND lg_ort = _lgo_name
                AND lg_chnr  = TRIM(_gebrstufe)
                -- AND lg_anztot = CAST(_bestand AS NUMERIC(12,4)) -- macht schon das UPDATE
                )
    ;

    UPDATE lagartikelkonf SET lgoa_melde = lgoa_melde + AsNumeric(_bestandMELDE, True),
                              lgoa_soll = lgoa_soll + AsNumeric(_bestandSOLL, True),
                              modified_by=_modified_by,
                              modified_date = current_date
        WHERE lgoa_aknr = TRIM(_matnr)
          AND lgoa_lgo_name = _lgo_name
    ;

    INSERT INTO lagartikelkonf (lgoa_aknr, lgoa_lgo_name, lgoa_melde, lgoa_soll, insert_by, insert_date)
        SELECT
          TRIM(_matnr),
          _lgo_name,
          AsNumeric(_bestandMELDE, True),
          AsNumeric(_bestandSOLL, True),
          _modified_by,
          current_date
        WHERE NOT EXISTS (SELECT 1 FROM lagartikelkonf
                WHERE lgoa_aknr = TRIM(_matnr)
                AND lgoa_lgo_name = _lgo_name
                )
    ;
 END $$ LANGUAGE plpgsql VOLATILE;
--
CREATE OR REPLACE FUNCTION TLager.lagerlog__lo_aktion__translate(
      _lo_aktion varchar
  )
  RETURNS varchar
  AS $$

      SELECT
        CASE _lo_aktion
            WHEN '+'    THEN  lang_text(29336)    -- Lagerzugang
            WHEN '=+'   THEN  lang_text(29337)    -- Korrektur Lagerzugang
            WHEN '-'    THEN  lang_text(29338)    -- Lagerabgang
            WHEN '=-'   THEN  lang_text(29339)    -- Lagerabgang wurde bearbeitet
            WHEN '++'   THEN  lang_text(29340)    -- Lagerzugang durch permanente Inventur (neuer Lagerort)
            WHEN '=='   THEN  lang_text(29341)    -- Korrektur durch permanente Inventur (bestehender Lagerort wurde geändert)
            WHEN '--'   THEN  lang_text(29342)    -- Lagerabgang durch permanente Inventur (bestehender Lagerort wurde gelöscht)
            WHEN '==+'  THEN  lang_text(29343)    -- Interne Lagerbewegung durch LZ > Erhöhung lagernd total
            WHEN '==-'  THEN  lang_text(29344)    -- Interne Lagerbewegung durch LA > Verringerung lagernd total
            WHEN '=+!'  THEN  lang_text(29345)    -- Korrektur, nicht bestandsverändert
            WHEN '=-!'  THEN  lang_text(29346)    -- Korrektur, nicht bestandsverändert
            WHEN '==|'  THEN  lang_text(29347)    -- Mengenänderung durch Abschluss der Stichtagsinventur
            WHEN '++|'  THEN  lang_text(29348)    -- Lagerort neu angelegt durch Abschluss der Stichtagsinventur
            WHEN '--|'  THEN  lang_text(29349)    -- Lagerort wurde durch Abschluss der Stichtagsinventur gelöscht
            WHEN '?=|'  THEN  lang_text(29350)    -- Lagerort, Charge oder Artikel wurden durch Abschluss der Stichtagsinventur umgeschrieben

            ELSE              '???'
        END

  $$ LANGUAGE sql STABLE;


--- #14794 Zentrale Funktion zum Erstellen Lagerlog: lag
CREATE OR REPLACE FUNCTION TLager.lagerlog__create(
    aktion      varchar,
    newData     lag,
    oldData     lag DEFAULT NULL
    )
    RETURNS     void
    AS $$
    DECLARE _isuebuchung boolean = false;
    BEGIN
        IF newData IS NULL THEN                  --- DELETE

           -- Lagerbewegung UE: kein Lagerlog-Eintrag -- https://redmine.prodat-sql.de/issues/15469
           IF (SELECT w_a2_id IS NOT null FROM wendat WHERE w_wen = oldData.lg_w_wen) THEN
              _isuebuchung := true;
              -- keine Lagerlog bei UE Buchungen. Sonst müßte alles mögliche (Stichtagsinventur usw) angepasst werden um diese Datensätze zu ignorieren
              RETURN;
           END IF;

           INSERT INTO lagerlog (
              lo_aktion,
              lo_aknr_new,
              lo_stk_new,
              lo_stk_old,
              lo_lgort_new,
              lo_lgchnr_new,
              lo_dim1_new,
              lo_dim2_new,
              lo_dim3_new,
              lo_txt
              )
           VALUES (
              aktion,              --- lo_aktion,
              oldData.lg_aknr,     --- lo_aknr_new,
              0,                   --- lo_stk_new,
              oldData.lg_anztot,   --- lo_stk_old,
              oldData.lg_ort,      --- lo_lgort_new,
              oldData.lg_chnr,     --- lo_lgchnr_new,
              oldData.lg_dim1,     --- lo_dim1_new,
              oldData.lg_dim2,     --- lo_dim2_new,
              oldData.lg_dim3,     --- lo_dim3_new,
              'DELETE'             --- lo_txt
              );
        ELSE                                     --- INSERT / UPDATE

           -- Lagerbewegung UE: kein Lagerlog-Eintrag -- https://redmine.prodat-sql.de/issues/15469
           IF (SELECT w_a2_id IS NOT null FROM wendat WHERE w_wen = newData.lg_w_wen) THEN
              _isuebuchung := true;
              -- keine Lagerlog bei UE Buchungen. Sonst müßte alles mögliche (Stichtagsinventur usw) angepasst werden um diese Datensätze zu ignorieren
              RETURN;
           END IF;

           INSERT INTO lagerlog (
              lo_aktion,
              lo_aknr_new,
              lo_aknr_old,
              lo_stk_new,
              lo_stk_old,
              lo_lgort_new,
              lo_lgort_old,
              lo_lgchnr_new,
              lo_lgchnr_old,
              lo_dim1_new,
              lo_dim1_old,
              lo_dim2_new,
              lo_dim2_old,
              lo_dim3_new,
              lo_dim3_old,
              lo_txt
              )
           VALUES          (
              aktion,              --- lo_aktion,
              newData.lg_aknr,     --- lo_aknr_new,
              oldData.lg_aknr,     --- lo_aknr_old,
              newData.lg_anztot,   --- lo_stk_new,
              oldData.lg_anztot,   --- lo_stk_old,
              newData.lg_ort,      --- lo_lgort_new,
              oldData.lg_ort,      --- lo_lgort_old,
              newData.lg_chnr,     --- lo_lgchnr_new,
              oldData.lg_chnr,     --- lo_lgchnr_old,
              newData.lg_dim1,     --- lo_dim1_new,
              oldData.lg_dim1,     --- lo_dim1_old,
              newData.lg_dim2,     --- lo_dim2_new,
              oldData.lg_dim2,     --- lo_dim2_old,
              newData.lg_dim3,     --- lo_dim3_new,
              oldData.lg_dim3,     --- lo_dim3_old,
              newData.lg_txt       --- lo_txt
              );
        END IF;
    END $$ LANGUAGE plpgsql;
--
--- #14794 Zentrale Funktion zum Erstellen Lagerlog: wendat
CREATE OR REPLACE FUNCTION TLager.lagerlog__create(
    aktion      varchar,
    newData     wendat,
    oldData     wendat DEFAULT NULL
    )
    RETURNS void
    AS $$
    DECLARE _isuebuchung boolean = false;
    BEGIN
        IF newData IS NULL THEN    --- DELETE

           -- Lagerbewegung UE: kein Lagerlog-Eintrag -- https://redmine.prodat-sql.de/issues/15469
           IF oldData.w_a2_id IS NOT null THEN
              _isuebuchung := true;
              -- keine Lagerlog bei UE Buchungen. Sonst müßte alles mögliche (Stichtagsinventur usw) angepasst werden um diese Datensätze zu ignorieren
              RETURN;
           END IF;

           INSERT INTO lagerlog (
              lo_aktion,
              lo_wendat,
              lo_aknr_new,
              lo_stk_new,
              lo_stk_old,
              lo_dim1_new,
              lo_dim2_new,
              lo_dim3_new,
              lo_lgort_new,
              lo_lgchnr_new,
              lo_txt
              )
           VALUES (
              aktion,                  --- llo_aktion,
              oldData.w_wen,           --- llo_wendat,
              oldData.w_aknr,          --- llo_aknr_new,
              0,                       --- llo_stk_new,
              oldData.w_zugang_uf1,    --- llo_stk_old,
              oldData.w_dim1,          --- llo_dim1_new,
              oldData.w_dim2,          --- llo_dim2_new,
              oldData.w_dim3,          --- llo_dim3_new,
              oldData.w_lgort,         --- llo_lgort_new,
              oldData.w_lgchnr,        --- llo_lgchnr_new,
              'DELETE'                 --- llo_txt
              );
        ELSE     --- INSERT / UPDATE

           -- Lagerbewegung UE: kein Lagerlog-Eintrag -- https://redmine.prodat-sql.de/issues/15469
           IF newData.w_a2_id IS NOT null THEN
              _isuebuchung := true;
              -- keine Lagerlog bei UE Buchungen. Sonst müßte alles mögliche (Stichtagsinventur usw) angepasst werden um diese Datensätze zu ignorieren
              RETURN;
           END IF;

           INSERT INTO lagerlog (
              lo_aktion,
              lo_wendat,
              lo_aknr_new,
              lo_aknr_old,
              lo_stk_new,
              lo_stk_old,
              lo_dim1_new,
              lo_dim1_old,
              lo_dim2_new,
              lo_dim2_old,
              lo_dim3_new,
              lo_dim3_old,
              lo_lgort_new,
              lo_lgort_old,
              lo_lgchnr_new,
              lo_lgchnr_old,
              lo_txt
              )
            VALUES          (
              aktion,                   --- lo_aktion,
              newData.w_wen,            --- lo_wendat,
              newData.w_aknr,           --- lo_aknr_new,
              oldData.w_aknr,           --- lo_aknr_old,
              newData.w_zugang_uf1,     --- lo_stk_new,
              oldData.w_zugang_uf1,     --- lo_stk_old,
              newData.w_dim1,           --- lo_dim1_new,
              oldData.w_dim1,           --- lo_dim1_old,
              newData.w_dim2,           --- lo_dim2_new,
              oldData.w_dim2,           --- lo_dim2_old,
              newData.w_dim3,           --- lo_dim3_new,
              oldData.w_dim3,           --- lo_dim3_old,
              newData.w_lgort,          --- lo_lgort_new,
              oldData.w_lgort,          --- lo_lgort_old,
              newData.w_lgchnr,         --- lo_lgchnr_new,
              oldData.w_lgchnr,         --- lo_lgchnr_old,
              newData.w_lbt             --- lo_txt
            );
        END IF;
    END $$ LANGUAGE plpgsql;
--
--- #14794 Zentrale Funktion zum Erstellen Lagerlog: lifsch
CREATE OR REPLACE FUNCTION TLager.lagerlog__create(
    aktion      varchar,
    newData     lifsch,
    oldData     lifsch DEFAULT NULL
    )
    RETURNS void
    AS $$
    DECLARE _isuebuchung boolean = false;
    BEGIN
        IF newData IS NULL THEN     --- DELETE

            -- Lagerbewegung UE: kein Lagerlog-Eintrag -- https://redmine.prodat-sql.de/issues/15469
            IF (SELECT TSystem.ENUM_GetValue(ag_stat, 'UE') FROM auftg WHERE ag_id = oldData.l_ag_id)
            THEN
               _isuebuchung := true;
              -- keine Lagerlog bei UE Buchungen. Sonst müßte alles mögliche (Stichtagsinventur usw) angepasst werden um diese Datensätze zu ignorieren
              RETURN;
            END IF;

            INSERT INTO lagerlog (
                lo_aktion,
                lo_lifsch,
                lo_aknr_new,
                lo_stk_new,
                lo_stk_old,
                lo_dim1_new,
                lo_dim2_new,
                lo_dim3_new,
                lo_lgort_new,
                lo_lgchnr_new,
                lo_txt
                )
            VALUES (
               aktion,                --- lo_aktion,
               oldData.l_nr,          --- lo_lifsch,
               oldData.l_aknr,        --- lo_aknr_new,
               0,                     --- lo_stk_new,
               oldData.l_abgg_uf1,    --- lo_stk_old,
               oldData.l_dim1,        --- lo_dim1_new,
               oldData.l_dim2,        --- lo_dim2_new,
               oldData.l_dim3,        --- lo_dim3_new,
               oldData.l_lgort,       --- lo_lgort_new,
               oldData.l_lgchnr,      --- lo_lgchnr_new,
               'DELETE'               --- lo_txt
               );
        ELSE    --- INSERT / UPDATE

            -- Lagerbewegung UE: kein Lagerlog-Eintrag -- https://redmine.prodat-sql.de/issues/15469
            IF (SELECT TSystem.ENUM_GetValue(ag_stat, 'UE') FROM auftg WHERE ag_id = newData.l_ag_id)
            THEN
               _isuebuchung := true;
              -- keine Lagerlog bei UE Buchungen. Sonst müßte alles mögliche (Stichtagsinventur usw) angepasst werden um diese Datensätze zu ignorieren
              RETURN;
            END IF;

            INSERT INTO lagerlog (
                lo_aktion,
                lo_lifsch,
                lo_aknr_new,
                lo_aknr_old,
                lo_stk_new,
                lo_stk_old,
                lo_dim1_new,
                lo_dim1_old,
                lo_dim2_new,
                lo_dim2_old,
                lo_dim3_new,
                lo_dim3_old,
                lo_lgort_new,
                lo_lgort_old,
                lo_lgchnr_new,
                lo_lgchnr_old,
                lo_txt
                )
            VALUES (
                aktion,                 --- lo_aktion,
                newData.l_nr,           --- lo_lifsch,
                newData.l_aknr,         --- lo_aknr_new,
                oldData.l_aknr,         --- lo_aknr_old,
                newData.l_abgg_uf1,     --- lo_stk_new,
                oldData.l_abgg_uf1,     --- lo_stk_old,
                newData.l_dim1,         --- lo_dim1_new,
                oldData.l_dim1,         --- lo_dim1_old,
                newData.l_dim2,         --- lo_dim2_new,
                oldData.l_dim2,         --- lo_dim2_old,
                newData.l_dim3,         --- lo_dim3_new,
                oldData.l_dim3,         --- lo_dim3_old,
                newData.l_lgort,        --- lo_lgort_new,
                oldData.l_lgort,        --- lo_lgort_old,
                newData.l_lgchnr,       --- lo_lgchnr_new,
                oldData.l_lgchnr,       --- lo_lgchnr_old,
                newData.l_azutx         --- lo_txt
               );
        END IF;
    END $$ LANGUAGE plpgsql;

-- Erstellen von Lager-Datensätzen auf Basis wendat/lifsch
CREATE OR REPLACE FUNCTION TLager.lag__from__wendat(_wendat wendat) RETURNS lag
    AS $$
        SELECT * FROM lag
         WHERE lg_aknr = _wendat.w_aknr
           AND lg_ort  = _wendat.w_lgort
           AND lg_chnr = _wendat.w_lgchnr
           AND lg_dim1 = _wendat.w_dim1
           AND lg_dim2 = _wendat.w_dim2
           AND lg_dim3 = _wendat.w_dim3;
    $$ LANGUAGE sql STABLE;


CREATE OR REPLACE FUNCTION TLager.lag__lg_id__from__wendat(_wendat wendat) RETURNS integer
    AS $$
        SELECT lg_id FROM TLager.lag__from__wendat(_wendat);
    $$ LANGUAGE sql STABLE;

CREATE OR REPLACE FUNCTION TLager.lag__from__wendat__create(
    IN _wendat          wendat,
    IN _negative        boolean DEFAULT false,
    IN _rck_w_wen       integer DEFAULT null,
    IN _rck_w_lds_id    integer DEFAULT null,
    IN _isuebuchung__sperr boolean DEFAULT false
    )
    RETURNS lag
    AS $$
        INSERT INTO lag
                    (lg_aknr, lg_ort, lg_chnr,
                     lg_dim1, lg_dim2, lg_dim3,
                     lg_anztot,
                     lg_w_wen,
                     lg_ld_id,
                     lg_konsDat, lg_konsEnde,
                     lg_txt,
                     lg_sperr
                     )
             VALUES (_wendat.w_aknr, _wendat.w_lgort, _wendat.w_lgchnr,
                     _wendat.w_dim1, _wendat.w_dim2, _wendat.w_dim3,
                     _wendat.w_zugang_uf1 * ifthen(_negative, -1, 1),
                     coalesce( _rck_w_wen,    _wendat.w_wen ),    --- lg_w_wen
                     coalesce( _rck_w_lds_id, _wendat.w_lds_id ), --- lg_ld_id
                     _wendat.w_konsDat, _wendat.w_konsEnde,
                     _wendat.w_lbt,
                     _isuebuchung__sperr
                     )
          RETURNING lag;
    $$ LANGUAGE sql;

CREATE OR REPLACE FUNCTION TLager.lag__from__lifsch(_lifsch lifsch) RETURNS lag
    AS $$
        SELECT * FROM lag
         WHERE lg_aknr = _lifsch.l_aknr
           AND lg_ort  = _lifsch.l_lgort
           AND lg_chnr = _lifsch.l_lgchnr
           AND lg_dim1 = _lifsch.l_dim1
           AND lg_dim2 = _lifsch.l_dim2
           AND lg_dim3 = _lifsch.l_dim3;
    $$ LANGUAGE sql STABLE;

CREATE OR REPLACE FUNCTION TLager.lag__lg_id__from__lifsch(_lifsch lifsch) RETURNS integer
    AS $$
        SELECT lg_id FROM TLager.lag__from__lifsch(_lifsch);
    $$ LANGUAGE sql STABLE;


CREATE OR REPLACE FUNCTION TLager.lag__from__lifsch__create(
    IN _lifsch      lifsch,
    IN _negative    boolean DEFAULT false,
    IN _l_konsDat   date    DEFAULT null,
    IN _l_konsEnde  date    DEFAULT null
    )
    RETURNS lag
    AS $$
        INSERT INTO lag
                    (lg_aknr, lg_ort, lg_chnr,
                     lg_dim1, lg_dim2, lg_dim3,
                     lg_anztot,
                     lg_w_wen,
                     lg_ld_id,
                     lg_konsDat, lg_konsEnde
                     )
             VALUES (_lifsch.l_aknr, _lifsch.l_lgort, _lifsch.l_lgchnr,
                     _lifsch.l_dim1, _lifsch.l_dim2, _lifsch.l_dim3,
                     _lifsch.l_abgg_uf1 * ifthen(_negative, -1, 1),
                     _lifsch.l_w_wen,
                     _lifsch.l_ld_id,
                     _l_konsDat, _l_konsEnde
                     )
          RETURNING lag
    $$ LANGUAGE sql;


-- stkl für UE #4848
CREATE OR REPLACE FUNCTION  TLager.wendat__w_zugang__by__ld_id__sum__stichtag(
        _date date,
        _aknr varchar,
        _ldid integer
        ) RETURNS numeric
        AS $$
    DECLARE _r numeric(12,4);
    BEGIN
        SELECT sum(w_zugang_uf1) INTO _r
          FROM wendat
         WHERE w_aknr = aknr
           AND w_zug_dat::date <= _date
           AND w_lds_id = _ldid;
        RETURN _r;
    END $$ LANGUAGE plpgsql STABLE;

CREATE OR REPLACE FUNCTION TLager.wendat__w_zugang__by__ag_id__sum__stichtag(
        _date date,
        _aknr varchar,
        _agid integer
        ) RETURNS numeric
        AS $$
    DECLARE _r numeric(12,4);
    BEGIN
        SELECT sum(w_zugang_uf1) INTO _r
          FROM auftg
          JOIN ldsdok ON ag_id = ld_ag_id
          JOIN wendat ON w_lds_id = ld_id
         WHERE w_aknr = _aknr
           AND w_zug_dat::date <= _date
           AND ag_id = _agid;
        RETURN _r;
    END $$ LANGUAGE plpgsql STABLE;
--


-- Funktionen für Lagerorte, Lagerbereich
 -- #6259 Lagerort zusammenstezen > Lagerbereich | Lagerort
 --- #13472 generate_lagerort > > TLager.lag__lagort__bereich__generate umbenennen
CREATE OR REPLACE FUNCTION TLager.lag__lagort__bereich__generate(_lagbereich varchar, _lagort varchar) RETURNS varchar AS $$
    BEGIN
        IF coalesce(_lagbereich, '') <> '' AND coalesce(_lagort, '') = '' THEN --Lagerbereich OHNE Lagerort ist unzulässig, führt zu keinem Lagerort (wäre sonst ja nur "H1 | "
           _lagort := NULL;
        END IF;
        RETURN coalesce(nullif(_lagbereich, '') || ' | ', '') || _lagort;
    END $$ LANGUAGE plpgsql IMMUTABLE;


-- #6259 Lagerort splitten > Lagerbereich | Lagerort
 --- #13471 Funktion ist umbenannt split_lagerort > TLager.lag__lagort__bereich__split
CREATE OR REPLACE FUNCTION TLager.lag__lagort__bereich__split(IN _lagerort varchar, OUT lgort varchar, OUT lagort varchar, OUT lagbereich varchar) RETURNS record AS $$
    DECLARE  _SplitPos integer;
    BEGIN
        -- wenn in lg_lagort bereits der zusammengesetzte lagerort eingetragen wird, die felder auseinandernehmen
        _SplitPos := position(' | ' in _lagerort);
        lgort     := _lagerort; -- H2 | MON
        lagort    := _lagerort;
        IF _SplitPos > 0 THEN
           lagbereich := substring(_lagerort, 1, _SplitPos - 1); -- in Lagbereich "H2" übernehmen aus "H2 | MON"
           lagort     := substring(_lagerort, _SplitPos + 3, char_length(_lagerort) - _SplitPos - 3 + 1); -- MON aus "H2 | MON"
        END IF;
        --
        RETURN;
    END $$ LANGUAGE plpgsql IMMUTABLE;
/* Funktonen sind Entwurf und nicht wirklich geprüft. Es fehlen mMn (DS) Fälle, zB bei wendat delete gibt es nur ein insert und kein update
--- #14794 -> #14859  Zentrale Funktion für INSERT INTO lag
CREATE OR REPLACE FUNCTION TLager.lag__upsert(
    IN _newData             wendat,
    IN _oldData             wendat   DEFAULT null,
    IN _rck_w_wen           integer  DEFAULT null,
    IN _rck_w_lds_id        integer  DEFAULT null,
    IN _isuebuchung__sperr  boolean  DEFAULT false
    )
    RETURNS VOID
    AS $$
    DECLARE
        newData_true        boolean;
        oldData_true        boolean;
        _lgid               integer := 0;
        _lgwwen             integer;
        _lgldid             integer;
    BEGIN

        IF _newData IS null THEN
            newData_true := false;
        ELSE
            newData_true := true;
        END IF;

        IF _oldData IS null THEN
            oldData_true := false;
        ELSE
            oldData_true := true;
        END IF;

        IF    ( newData_true = true )  AND ( oldData_true = false ) THEN    ---  Var.1 wendat__a_10_i(), Var.2 wendat__a_14_u() neuen Lagerort um neuen Zugang erhöhen

            SELECT coalesce( TLager.lag__lg_id__from__wendat(_newData), 0) INTO _lgid;

            IF _lgid = 0 THEN
                SELECT (TLager.lag__from__wendat__create(_newData, false, _rck_w_wen, _rck_w_lds_id, _isuebuchung__sperr)).lg_id
                  INTO _lgid;
            ELSE
                UPDATE lag
                   SET lg_anztot   = lg_anztot + _newData.w_zugang_uf1,
                       lg_sperr    = lgort_dosperr( _newData.w_lgort ),
                       lg_w_wen    = CASE WHEN lg_anztot <= 0 THEN coalesce( _rck_w_wen, _newData.w_wen )       ELSE lg_w_wen                                            END, -- Bezug erneut herstellen bei Bestand <=0 (Standardlagerort)
                       lg_ld_id    = CASE WHEN lg_anztot <= 0 THEN coalesce( _rck_w_lds_id, _newData.w_lds_id ) ELSE lg_ld_id                                            END, -- ebenso
                       lg_konsdat  = CASE WHEN lg_anztot <= 0 THEN _newData.w_konsdat                           ELSE least( lg_konsdat, _newData.w_konsdat )             END,
                       lg_konsEnde = CASE WHEN lg_anztot <= 0 THEN _newData.w_konsende                          ELSE least( lg_konsende, _newData.w_konsende )           END,
                       lg_txt      = CASE WHEN lg_txt IS null THEN _newData.w_lbt                               ELSE lg_txt || coalesce( E'\n\n' || _newData.w_lbt, '' ) END
                 WHERE lg_id = _lgid
                 RETURNING
                       lg_w_wen, lg_ld_id
                  INTO _lgwwen, _lgldid;

                -- Zusammenlegen von Lagerorten mit Bezug zu WE oder Bestellung ggf entfernen
                IF _lgwwen IS DISTINCT FROM coalesce( _rck_w_wen, _newData.w_wen ) AND _lgwwen IS NOT null THEN
                   UPDATE lag SET lg_w_wen = null WHERE lg_id = _lgid;
                   PERFORM prodat_hint( langtext( 21283 ) ); -- Zusammenlegen von Lagerbeständen hat Wareneingangsbezug entfernt
                END IF;
                IF _lgldid IS DISTINCT FROM coalesce( _rck_w_lds_id, _newData.w_lds_id ) AND _lgldid IS NOT null THEN
                   UPDATE lag SET lg_ld_id = NULL WHERE lg_id = _lgid;
                   PERFORM prodat_hint( langtext( 21284 ) ); -- Zusammenlegen von Lagerbeständen hat Bestellbezug entfernt
                END IF;
            END IF;
            RAISE NOTICE 'Var.1 wendat__a_10_i(), Var.2 wendat__a_14_u()  lg_id = %', _lgid;

        ELSIF ( newData_true = true )  AND ( oldData_true = true ) THEN    ---   Var.3 wendat__a_14_u() -- alten Lagerort um alten Zugang verringern

            SELECT coalesce( TLager.lag__lg_id__from__wendat(_oldData), 0) INTO _lgid;

            IF _lgid = 0 AND NOT TSystem.Settings__GetBool( 'no_neg_lag' ) THEN
                INSERT INTO lag (
                  lg_aknr,
                  lg_ort,
                  lg_chnr,
                  lg_dim1,
                  lg_dim2,
                  lg_dim3,
                  lg_anztot,
                  lg_w_wen,
                  lg_ld_id,
                  lg_konsDat,
                  lg_konsEnde,
                  lg_txt
                )
                VALUES (
                  _oldData.w_aknr,                             --- lg_aknr
                  _oldData.w_lgort,                            --- lg_ort
                  _oldData.w_lgchnr,                           --- lg_chnr
                  _oldData.w_dim1,                             --- lg_dim1
                  _oldData.w_dim2,                             --- lg_dim2
                  _oldData.w_dim3,                             --- lg_dim3
                  - _oldData.w_zugang_uf1,                     --- lg_anztot

                  -- !!!!! ACHTUNG -> NEWDATA!!!! WIESO?!
                  _newData.w_wen,                              --- lg_w_wen
                  _newData.w_lds_id,                           --- lg_ld_id
                  _newData.w_konsDat,                          --- lg_konsDat
                  _newData.w_konsEnde,                         --- lg_konsEnde
                  _newData.w_lbt                               --- lg_txt
                ) RETURNING lg_id INTO _lgid;
            ELSE
                -- alten Lagerort um alten Zugang verringern
                UPDATE lag
                   SET lg_anztot = lg_anztot - _oldData.w_zugang_uf1
                 WHERE lg_id = _lgid;
            END IF;
            RAISE NOTICE 'Var.3 wendat__a_14_u()  lg_id = %', _lgid;
        ELSIF ( newData_true = false ) AND ( oldData_true = true ) THEN    ---   Var.4 wendat__b_d()
            SELECT (TLager.lag__from__wendat__create(_oldData, true, null, null, _isuebuchung__sperr)).lg_id
                  INTO _lgid;
            RAISE NOTICE 'Var.4 wendat__b_d()  lg_id = %', _lgid;
        END IF;

    END $$ LANGUAGE plpgsql;
---
CREATE OR REPLACE FUNCTION TLager.lag__upsert(
    IN _newData                lifsch,
    IN _oldData                lifsch,
    IN _l_konsDat              DATE,
    IN _l_konsEnde             DATE,
    IN _tg_op                  VARCHAR
    )
    RETURNS VOID
    AS $$
    DECLARE
        newData_true    BOOLEAN;
        oldData_true    BOOLEAN;
        lgid            INTEGER := 0;
        lgwwen          INTEGER;
        lgldid          INTEGER;
    BEGIN
        IF _newData IS null THEN
            newData_true := false;
        ELSE
            newData_true := true;
        END IF;

        IF _oldData IS null THEN
            oldData_true := false;
        ELSE
            oldData_true := true;
        END IF;

        IF    ( newData_true = true )  AND ( oldData_true = false ) THEN    ---  Var.1-2 wendat__a_10_i(), Var.2 wendat__a_14_u() neuen Lagerort um neuen Zugang erhöhen

            SELECT coalesce( TLager.lag__lg_id__from__lifsch(_newData), 0 ) INTO lgid;

            IF lgid = 0 THEN
                SELECT (TLager.lag__from__lifsch__create(_newData, true)).lg_id
                  INTO lgid;

            ELSE
                IF _tg_op = 'UPDATE' THEN
                    UPDATE lag SET lg_anztot = lg_anztot - _newData.l_abgg_uf1 WHERE lg_id = lgid;
                END IF;

            END IF;

            RAISE NOTICE 'Var. 1-2  lg_id = %', lgid;

        ELSIF ( newData_true = false ) AND ( oldData_true = true ) THEN    ---   Var.3-4 wendat__b_d()

            SELECT coalesce( TLager.lag__lg_id__from__lifsch(_oldData), 0 ) INTO lgid;

            IF lgid = 0 THEN
                SELECT (TLager.lag__from__lifsch__create(_newData, false, _l_konsDat, _l_konsEnde)).lg_id
                  INTO lgid;
            ELSE
                UPDATE lag
                   SET lg_anztot = lg_anztot + _oldData.l_abgg_uf1
                 WHERE lg_id = pgid;
            END IF;

            RAISE NOTICE 'Var. 3-4  lg_id = %', lgid;

        END IF;

    END $$ LANGUAGE plpgsql;
*/

-- Sonderlösungen und Custom Funktionen

   -- Dummy-Funktion, welche im Customer überschrieben wird. (LOLL). Suche nach Funktionsnamen in Quellen (TotalCustomer.SQL)
    CREATE OR REPLACE FUNCTION z_50_customer.TFormLagerAbSofort__ab_ix__by__agnr__customfunction(ident VARCHAR) RETURNS INTEGER AS $$
     BEGIN
      RETURN tplanterm.auftg_get_abk(ag_id, false) FROM auftg WHERE ag_astat = 'E' AND ag_nr = ident AND NOT ag_done ORDER BY ag_pos LIMIT 1; -- Auftragsnummer Airbus, welche über Barcode vom Airbus-Dokument kommt
     END $$ LANGUAGE plpgsql;

--


CREATE OR REPLACE FUNCTION TLager.mapsernr__ms_ids__by__ld_id__get( _ld_id integer )
    RETURNS SETOF integer
    AS $$

        -- ermittelt zu einer Bestellung die zugeordneten Vorgabeseriennummern

        -- Vorgabseriennummern
        SELECT ms_id
          FROM mapsernr
         WHERE ms_pkey::INTEGER = _ld_id
           AND ms_table = 'ldsdok'::REGCLASS

         UNION

        -- zugebuchte Seriennummern
        SELECT ms_id
          FROM mapsernr
          JOIN lagsernr ON lgs_id = ms_lgs_id
          -- dem Wareneingang zugewiesene Seriennummern
          JOIN wendat ON w_wen = lgs_w_wen
         WHERE w_lds_id = _ld_id;

    $$ LANGUAGE sql STABLE;


-- Seriennummern welche dem Einkauf bereits zugewiesen waren: vorgabeseriennummern
CREATE OR REPLACE FUNCTION TLager.lagsernr__lgs_ids__from__ldsdok__by__w_wen__get(
    IN  _w_wen integer,
    IN _unassigned_only boolean DEFAULT true
    )
    RETURNS SETOF integer
    AS $$

      -- ermittelt zu einem gegebenen LZ alle zur zugehörigen Bestellung angelegten Vorgabeseriennummern
      -- (d.h. ihre IDs), welche ihrerseits noch keinem LZ zugeordnet sind

      SELECT ms_lgs_id
        FROM wendat
             -- der zum Wareneingang zugehörigen Bestellung zugeordneten Produktseriennummern
        JOIN mapsernr ON ms_pkey::INTEGER = w_lds_id AND ms_table = 'ldsdok'::REGCLASS
             -- Join brauchen wir nur, um zugeordnete rauszufiltern (lgs_w_wen null)
        JOIN lagsernr     ON lgs_id = ms_lgs_id AND (lgs_w_wen IS null OR NOT _unassigned_only)
       WHERE w_wen = _w_wen
       ORDER BY
             lgs_id;

    $$ LANGUAGE sql STABLE;

CREATE OR REPLACE FUNCTION TLager.lagsernr__lgs_ids__by__w_wen__get(
    IN _w_wen integer,
       -- Seriennummern welche über die Vorgabe im Einkauf mittels der w_wen Bezug zum Wareneingang haben
       -- (Anwendungsfall: Zuweisen von Seriennummern. Entweder direkt durch DB erstellte und em WE zugordnet, oder Auswahl aus den Vorgaben)
    IN _ldsdok_presettings boolean DEFAULT false,
       -- nur freie Vorgabeseriennummern -> Seriennummern die noch keinem WE zugeordnet sind
    IN _ldsdok_presettings_unassigend_only boolean DEFAULT true
    )
    RETURNS SETOF integer
    AS $$
        -- dem WE direkt zugewiesene Seriennummern
        SELECT lgs_id
          FROM lagsernr
         WHERE lgs_w_wen = _w_wen
         UNION
        -- Mögliche Seriennummern, die über WE bezug ermittelt werden aus Einkauf und dem WE zugewiesen werden könnten
        SELECT lagsernr__lgs_ids__from__ldsdok__by__w_wen__get
          FROM TLager.lagsernr__lgs_ids__from__ldsdok__by__w_wen__get(_w_wen, _ldsdok_presettings_unassigend_only)
         WHERE _ldsdok_presettings
         ORDER BY
               lgs_id;

    $$ LANGUAGE sql STABLE;

CREATE OR REPLACE FUNCTION TLager.lagsernr__lgs_ids__by__ld_id__get(
    IN _ld_id integer
       -- Bestellung an der die Vorgabe-Seriennummern hängen
    )
    RETURNS SETOF integer
    AS $$
        -- ermittelt zu einer Bestellung die zugeordneten Vorgabeseriennummern
        SELECT lgs_id FROM lagsernr
        JOIN mapsernr ON ms_id IN (SELECT TLager.mapsernr__ms_ids__by__ld_id__get( _ld_id )) AND ms_lgs_id = lgs_id;

    $$ LANGUAGE sql STABLE;

--

-- kundenspezifische Seriennummerngenerator
CREATE OR REPLACE FUNCTION z_50_customer.customer_lagsernr__presetting__create( _ld_id integer, _count integer, _w_wen integer )
RETURNS SETOF varchar AS $$
DECLARE
  _r record;
  x2e boolean;
BEGIN

  -- handelt es sich hier um eine x2e-Datenbank?
  x2e := EXISTS(
    SELECT 1 FROM information_schema.tables
    WHERE
            table_name = 'code_product_variant_revisions'
        AND table_schema = 'z_50_customer'
  );

  -- implementiert kundenspezifische Seriennummenrgeneratoren

  IF EXISTS( SELECT 1 FROM information_schema.schemata WHERE schema_name = 'z_50_customer' ) THEN

    -- x2e
    IF x2e THEN
      FOR _r IN ( SELECT z_50_customer.x2e__lagsernr__presetting__create( _ld_id, _count, _w_wen )) LOOP
        RETURN NEXT _r;
      END LOOP;
    END IF;

    -- SPEKTRA
    -- kommt zu gegebener Zeit

  END IF;

END $$ LANGUAGE plpgsql STABLE;
--

-- ermittelt die nächsten laufenden Seriennummern zu einer gegebenen Artikelnummer
--   Format: ARTNR#dddddd z.B. AW.EXTERN#000123
CREATE OR REPLACE FUNCTION tlager.sernr__ak_nr__create( _ak_nr varchar, _num integer ) RETURNS SETOF varchar AS $$
-- _ak_nr - Artikelnummer
-- _num   - Anzahl der zu generierenden Seriennummern
DECLARE
  _digit_number integer;
  _aknr varchar;
  _len integer;
  _max_sn integer;
  _sn varchar;
  _vorhanden boolean;
  _counter integer;
BEGIN

  -- Maximal 999.999 Seriennummern pro Artikel
  _digit_number := 6;

  -- Artikelnummern von Anhängseln befreien
  _aknr := tartikel.art__ak_nr__index( _ak_nr );
  IF _aknr IS null THEN
    RETURN;
  END IF;

  -- Länge der Artikelnummer + "#"
  _len := length( _aknr ) + 1;

  -- Größte, zum Artikel passende, vorhandene laufende Nummer ermitteln
  _max_sn := max( (substring( lgs_sernr FROM _len+1 FOR _digit_number ))::integer )
             FROM lagsernr
             WHERE lgs_sernr ~ ( '^' || _aknr || '#' || '[0-9](' || _digit_number || ')$' );
               -- '^ARTNR#[0-9](6)$'
  IF _max_sn IS null THEN
    _max_sn := 0;
  END IF;

  -- Schleife zur Ausgabe der zu generierenden Artikelnummern
  _counter := 0;
  LOOP
    _max_sn := _max_sn + 1;

    -- Schleie verlassen, wenn genug Seriennummern oder Limit erreicht
    EXIT WHEN _counter >= _num OR _max_sn >= 1000000;

    -- Seriennummer bauen
    _sn := concat( (_aknr || '#'), lpad( _max_sn, _digit_number, '0' ) );

    -- Absicherung geben doppelte Seriennumemrn
    _vorhanden := EXISTS( SELECT 1 FROM lagsernr WHERE lgs_sernr = _sn );
    IF NOT _vorhanden THEN
      _counter := _counter + 1;
      RETURN NEXT _sn;
    END IF;
  END LOOP;

END $$ LANGUAGE plpgsql STABLE STRICT;
--

-- Generierung von Seriennummern nach https://redmine.prodat-sql.de/issues/16985
-- Muster der Seriennummern #11118, #11173
CREATE OR REPLACE FUNCTION tlager.lagsernr__presetting__create( _ld_id integer, _count integer, _w_wen integer DEFAULT null ) RETURNS SETOF VARCHAR AS $$

  -- alle vorhandenen Vorgebeseriennummer zur Bestellposition, welche keinem LZ zugeordnet sind
  WITH

    -- nicht zugewiesene, noch vorhandene Vorgabeseriennummern
    ld_sernr AS (
      SELECT lgs_sernr
      FROM lagsernr
      JOIN mapsernr ON     ms_lgs_id = lgs_id
                       AND ms_pkey::INTEGER = _ld_id
                       AND ms_table = 'ldsdok'::REGCLASS
      WHERE lgs_w_wen IS null -- Nur nicht zugewiesene SN ausgeben
    ),

    -- Artikelnummer der Bestellung
    ld_data AS (
        SELECT ld_aknr AS aknr FROM ldsdok WHERE ld_id = _ld_id
    ),

    -- Artikelformat beim Wareneingang (true) oder das Standardformat mit führendem ~~ (false)
    sn_vorgabe AS (
        SELECT ac_sn_vorgabe FROM artcod
        JOIN art ON ak_ac = ac_n
        JOIN ld_data ON ld_data.aknr = ak_nr
    ),

    -- die Anzahl der zurückzugebenen Seriennummer wird bestimmt:
    -- entweder der übegebene Parameter _count oder die noch nicht eingelagerte Bestellmenge
    anzahl AS (
      SELECT max(
          ( SELECT count(*)::integer FROM ld_sernr ),
            coalesce( _count, ( SELECT ld_stk - ld_stkl FROM ldsdok WHERE ld_id = _ld_id ), 0 )
      ) AS anz
    ),

    -- erster Teil einer Seriennummer im Standardformat
    ident AS (
        SELECT coalesce( _w_wen :: varchar, -- Bei WE
                      ( SELECT aknr FROM ld_data ), -- Bei Vorgabeseriennummer (im Einkauf)
                        to_char( current_date, 'YYYYMMDD' ) -- Fallback auf heutiges Datum
       ) AS id
    ),

    -- kundenbezogenes Seriennummernformat
    customer_sernr AS (
        SELECT customer_lagsernr__presetting__create AS lgs_sernr
        FROM z_50_customer.customer_lagsernr__presetting__create( _ld_id, ( SELECT anz::integer FROM anzahl ) - ( SELECT count(*)::integer FROM ld_sernr ), _w_wen )
        CROSS JOIN sn_vorgabe
        WHERE _w_wen IS null OR ac_sn_vorgabe
    ),

    -- artikelbezogenens Seriennummernformat
    default_sernr AS (
        SELECT tlager.sernr__ak_nr__create( aknr, ( SELECT anz::integer FROM anzahl ) - ( SELECT count(*)::integer FROM ld_sernr ) ) AS lgs_sernr
        FROM ld_data
        CROSS JOIN sn_vorgabe
        WHERE _w_wen IS null OR ac_sn_vorgabe
    )

  SELECT lgs_sernr FROM (

    -- alle vorhandenen Vorgabeseriennummern
    SELECT
      0 AS sort,
      lgs_sernr
    FROM ld_sernr


    UNION

    -- alle Kunden-Vorgabeseriennummern
    SELECT
      1 AS sort,
      lgs_sernr
    FROM customer_sernr

    UNION

    -- alle Default-Vorgabeseriennummern
    SELECT
      2 AS sort,
      lgs_sernr
    FROM default_sernr

    UNION

    -- Alle generierten Seriennummern im Standardformat bestehen aus Ident + einer laufenden Nummer
    -- Der Ident kann sein (siehe WITH Query 'ident'):
    --   1. die aktuelle WE-Nummer,
    --   2. Vorgabeseriennummer (Bestellung) oder
    --   3. das aktuelle Datum
    -- Hier prüfen wir, ob ausreichend Seriennummern generiert waren. Im Fall nein, zB es waren 10 bestellt aber 12 Lagzugang werden 2 zusätzliche erstellt.

    SELECT
      3 AS sort,
      '~~' || ident || ' #' || lpad((start_nr + generate_series( 1, ( SELECT anz FROM anzahl )))::varchar, 3, '0' )
    FROM (
           SELECT
              id AS ident,
              CASE WHEN _w_wen IS NULL
                                -- Abfrage der maximalen laufenden Nummer (nicht bei WE)
                THEN COALESCE(( SELECT
                                  RIGHT(lgs_sernr, LENGTH(lgs_sernr) - LENGTH('~~'|| id ||' #'))::INT -- Extrahieren der laufenden Nummer
                                FROM lagsernr
                                WHERE lgs_sernr LIKE '~~'|| id ||' #%'
                                ORDER BY lgs_sernr DESC
                                LIMIT 1 ),
                               0)
                ELSE 0
                END AS start_nr
           FROM ident
         ) AS q1

    EXCEPT

    -- aussortiert wird alles was bereits mit einem passenden WE in Verbindung gebracht wird
    SELECT generate_series AS sort, lgs_sernr FROM generate_series( 1, 2 ), lagsernr WHERE
           lgs_w_wen = _w_wen
        OR lgs_w_wen IN (SELECT w_wen FROM wendat WHERE w_lds_id = _ld_id )
  ) AS x
  -- Sortierung: Vorgabeseriennummern haben Priorität
  ORDER BY sort ASC, lgs_sernr ASC
  -- Anzahl der Seriennummern:
  -- Maximum aus Anzahl der Vorgabeseriennummern, Bestellmenge und WE-Menge (soweit übergeben)
  LIMIT ( SELECT anz FROM anzahl )

$$ LANGUAGE sql STABLE;
----


--- #18577, Hilfsfunktion Notiz pro SN
CREATE OR REPLACE FUNCTION tlager.mapsernr__lagsernr__get_note(
    IN _ab_ix integer,
    IN _Seriennummer varchar,

    OUT capt varchar,
    OUT txt  text
  ) RETURNS RECORD AS $$

 BEGIN
    SELECT
      get_record_note_id_txt( lagsernr.dbrid, NULL, true, false ) AS capt,
      get_record_note_id_txt( lagsernr.dbrid, NULL, false, true ) AS txt
    INTO capt, txt
    FROM lagsernr
    JOIN mapsernr ON ms_lgs_id = lgs_id
      AND ms_pkey = ( SELECT ab_ld_id FROM abk WHERE ab_ix = _ab_ix ) -- Vorgabeseriennummer (= Bestellung/Lagsernr-Bezug)
      AND ms_table = 'ldsdok'::REGCLASS
    WHERE lagsernr.lgs_sernr LIKE _Seriennummer
      AND get_record_has_note( lagsernr.dbrid )
    ;

    RETURN;

 END $$ LANGUAGE plpgsql STABLE STRICT;
----

-- #19163, Hilfsfunktion für das Anlegen einer Notiz für eine Seriennummer,
-- welche ggf. erst noch anzulegen ist
-- gibt zurück, ob die Operation erfolgreich war
CREATE OR REPLACE FUNCTION tlager.lagsernr__note__create(
    _ab_ix integer,
    _sernr varchar,
    _betreff varchar,
    _kommentar text,
    _a2_n integer DEFAULT null
  ) RETURNS boolean AS $$
  DECLARE
    _ld_id integer;
    _lgs_id integer;
    _lgs_dbrid varchar;
    _a2_id integer;
    _rc_allg1 varchar;
  BEGIN

    _ld_id := ab_ld_id FROM abk WHERE ab_ix = _ab_ix;

    -- Bestellung, Seriennummer und Betreff werden gebraucht
    IF _ld_id IS null OR _sernr IS null OR _betreff IS null THEN
      RETURN false;
    END IF;

    -- bestehende Vorgabeseriennummer ermitteln
    _lgs_id :=
      lgs_id FROM lagsernr
      JOIN mapsernr ON ms_lgs_id = lgs_id AND ms_table = 'ldsdok'::REGCLASS AND ms_pkey = _ld_id
      WHERE lgs_sernr = _sernr;

    -- Keine gefunden? Dann neue Vorgabeseriennummer anlegen.
    IF _lgs_id IS null THEN
      _lgs_id := tlager.lagsernr__vorgabesernr__create( _sernr, _ld_id );
    END IF;

    -- Kommentar zur Vorgabeseriennummer schreiben
    IF _lgs_id IS NOT null THEN
      _lgs_dbrid := dbrid FROM lagsernr WHERE lgs_id = _lgs_id;

      -- ggf. JSON für AG bauen
      IF _a2_n IS NOT null THEN
        _a2_id := a2_id FROM ab2 WHERE a2_ab_ix = _ab_ix AND a2_n = _a2_n;
        IF _a2_id IS NOT null THEN
          _rc_allg1 := row_to_json(r)::varchar FROM ( SELECT _a2_id AS a2_id, _a2_n AS a2_n ) AS r;
        END IF;
      END IF;

      -- Kommentar speichern, AG-Infos als JSON in Feld rc_allg1
      INSERT INTO recnocomments
        ( rc_tablename, rc_dbrid  , rc_betreff,  rc_text  , rc_allg1  )
      VALUES
        ( 'lagsernr'  , _lgs_dbrid, _betreff  , _kommentar, _rc_allg1 );

      RETURN true;
    ELSE
      RETURN false;
    END IF;

  END $$ LANGUAGE plpgsql VOLATILE;
--

-- #20353 Spezialtyp für Suchoptionen, ein String reicht dafür nicht mehr aus
CREATE TYPE tabk.lagab_abk_suchoption AS (
    suchoption varchar,
    vorzug_bedarfsverursacher_gebunden boolean
);

-- #18078, 19489, 20082, 20353 berechnet die vorgesehen Abgänge eines ABK-bezogenne Lagerabgangs
CREATE OR REPLACE FUNCTION tabk.lagerabgang__abk__suchoption(
    IN  _suchoption tabk.lagab_abk_suchoption,
    OUT _mit_grundartikel boolean,
    OUT _mit_alternativartikel boolean,
    OUT _mit_bedarf boolean,
    OUT _grosse_lagermengen_bevorzugen boolean,
    OUT _ein_lagerort_pro_artikel boolean,
    OUT _vorzug_bedarfsverursacher_gebunden boolean
) RETURNS record AS $$

    DECLARE
      _so varchar;
    BEGIN

      _so := _suchoption.suchoption;
      IF _so IS null OR _so = '' THEN
        _so := 'ST';
      END IF;
      _vorzug_bedarfsverursacher_gebunden := coalesce( _suchoption.vorzug_bedarfsverursacher_gebunden, true );

      _mit_grundartikel := _so IN ( 'ST', 'GA', 'CH', 'AL', 'EI' );
      _mit_alternativartikel := _so IN ( 'GA', 'AL', 'AM' );
      _mit_bedarf := _so IN ( 'ST', 'GA', 'CH', 'EI' );
      _grosse_lagermengen_bevorzugen := _so IN ( 'CH' );
      _ein_lagerort_pro_artikel := _so IN ( 'EI' );
      RETURN;

END $$ LANGUAGE plpgsql IMMUTABLE;
--

CREATE OR REPLACE FUNCTION tabk.lg_ort__restmenge_kommission__is( _lg_ort varchar, _ab_ix integer )
RETURNS integer AS $$
DECLARE
  _tokens varchar[];
  _ks_lag varchar;
  _ks_abk varchar[];
  _andere_abk_offen boolean;
BEGIN

  -- Handelt es sich um einen Kommissionierlagerort?
  -- Rückgabe 0, nein
  -- Rückgabe null, wenn andere ABKs zum Lagerort noch offen sind, ansonsten
  --   Rückgabe -1, ja, aber Kostenstelle gehört nicht zur ABK
  --   Rückgabe 1, ja, und Kostenstelle gehört zur ABK

  IF coalesce( _lg_ort, '' ) !~ E'^KOMMISSION\_.+\_(\\*|[0-9]+)$' THEN
    RETURN 0;
  END IF;

  -- Gibt es noch andere offene ABKs zu diesem Lagerort?
  _andere_abk_offen :=
    NOT bool_and( ab_done ) FROM abk
    JOIN lag ON lg_txt = ab_ix::varchar AND ab_ix <> _ab_ix AND lg_ort = _lg_ort;

  -- Falls ja, dann null zurück.
  IF coalesce( _andere_abk_offen, false ) THEN
    RETURN null;
  END IF;

  -- Passt Kostenstelle zur ABK?
  _tokens := regexp_split_to_array( _lg_ort, E'\_' );
  _ks_lag := _tokens[2];
  _ks_abk := array_agg( DISTINCT a2_ks ) FROM ab2 WHERE a2_ab_ix = _ab_ix;

  -- Rückgabewert abhängig von dieser Zuordnung
  IF _ks_lag = ANY( _ks_abk ) THEN
    RETURN 1;
  ELSE
    RETURN -1;
  END IF;

END $$ LANGUAGE plpgsql STABLE;
--

-- #18078, berechnet die vorgesehen Abgänge eines ABK-bezogenen Lagerabgangs, eine Materiallistenposition
-- #18133, bevorzugt Kommissionslagerorte
-- #18489, Suchoption "möglichst wenige Chargen"
-- #20082, Suchoption "nur ein Lagerort pro Bedarfsposition"
-- #20353, Bevorzugung gebundener Bedarfsträger kann abgewählt werden
CREATE OR REPLACE FUNCTION tabk.auftg__lag__fillup(
      IN _auftg auftg,
      IN _abix integer,
      IN _menge_pro_abk numeric,
      IN _auftgmatinfo auftgmatinfo,
      IN _force_lgort varchar,
      IN _preferABK integer,
      IN _suchoption tabk.lagab_abk_suchoption = null
    )
    RETURNS VOID AS $$
    DECLARE
      _mit_grundartikel boolean;
      _mit_alternativartikel boolean;
      _mit_bedarf boolean;
      _grosse_lagermengen_bevorzugen boolean;
      _ein_lagerort_pro_artikel boolean;
      _vorzug_bedarfsverursacher_gebunden boolean;
      _bedarf numeric;
      _r integer;
      _aknr varchar;
      _lganztot_uf1 numeric;
      _datensatz boolean;
    BEGIN

      -- siehe auch https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Lager
      -- gibt abhängig von der Suchoption und davon, ob Alternativartikel zulässig sind,
      -- Lagerorte zur Bedarfsdeckung zurück. Hier die möglichen Suchoptionen
      --
      -- Standard ('ST' oder '' oder null): Standard-FIFO-Suche, welche nur Grundartikel berücksichtigt, bis zur Bedarfsdeckung
      -- Grund- vor Alternativartikel (GA): Standard-FIFO-Suche, es werden Alternativartikel erst dann in Betracht gezogen,
      --   wenn der Grundartikel nicht genügend Bestand hat
      -- Wenig Chargen (SH): Wie 'ST', nur werden hier große Lagermengen bevorzugt, um möglichst wenig verschiedene Chargen vorzuschlagen
      --   Chargen, die den Bedarf decken, werden weiterhin nach FIFO sortiert.
      -- Alle Artikel (AL): Es werden die Lagerorte aller zur Bedarfsdeckung infrage kommenden Artikel aufgelistet, ohne Rücksicht auf den Bedarf.
      -- Nur Alternativartikel (AM): Es werden die Lagerorte aller zur Bedarfsdeckung infrage kommenden Alternativartikel aufgelistet,
      --   ohne Rücksicht auf den Bedarf.

      SELECT * INTO _mit_grundartikel, _mit_alternativartikel, _mit_bedarf, _grosse_lagermengen_bevorzugen, _ein_lagerort_pro_artikel, _vorzug_bedarfsverursacher_gebunden
      FROM tabk.lagerabgang__abk__suchoption( _suchoption ) ;

      -- Bedarf ermitteln
      -- Ist keiner vorhanden, dann gibt es hier nichts zu tun.
      -- Bedarf ergibt sich aus dem Minimum der offenen Menge und der anteiligen Menge bei Teilbuchungen (DoMenge)
      _bedarf := me__menge_uf1__in__menge( _auftg.ag_mcv, min( _auftg.ag_stk_uf1 - _auftg.ag_stkl, _auftg.ag_stk_uf1 * _menge_pro_abk ) );
      IF _bedarf IS null THEN
        RETURN;
      END IF;

      _datensatz := false;
      FOR _r IN (
        SELECT
          lgid,

          -- die Reihenfolge der Lagerorte, die von tartikel.art__lag__lg_anztot__by__inputparams_all geliefet wird,
          -- als Sortierkriterium
          row_number() OVER () AS ord,

          -- #19649 bei Suchoption 'möglichst wenige Chargen' und Bedarfsdeckung möglichst kleine Chargen bevorzugen
          CASE
            WHEN _grosse_lagermengen_bevorzugen AND lganztot = _bedarf THEN lganztot
            ELSE null
          END AS sort_lagermenge_gleich_bedarf,

          -- #19649 bei Suchoption 'möglichst wenige Chargen' und Bedarfsdeckung möglichst kleine Chargen bevorzugen
          -- #20353 sollen die gebundenen Bedarfsverursacher keine Rolle spielen, dann FIFO-Reihenfolge
          CASE
            WHEN _grosse_lagermengen_bevorzugen AND lganztot > _bedarf AND _vorzug_bedarfsverursacher_gebunden THEN lganztot
            ELSE null
          END AS sort_lagermenge_groesser_bedarf,

          -- #19649 bei Suchoption 'möglichst wenige Chargen' und fehlender Bedarfsdeckung möglichst große Chargen bevorzugen
          -- #20353 sollen die gebundenen Bedarfsverursacher keine Rolle spielen, dann FIFO-Reihenfolge
          CASE
            WHEN _grosse_lagermengen_bevorzugen AND lganztot < _bedarf AND _vorzug_bedarfsverursacher_gebunden THEN -lganztot
            ELSE null
          END AS sort_lagermenge_kleiner_bedarf
        FROM tartikel.artoption_arts__aknr__get( _auftg.ag_aknr, _mit_alternativartikel )
        JOIN tartikel.art__lag__lg_anztot__by__inputparams_all(
            artoption_arts__aknr__get,
            _auftgmatinfo.agmi_beistell,
            ( SELECT v_stat = 'L' FROM versart WHERE v_id = _auftgmatinfo.agmi_v_id ),
            _auftg.ag_nr,
            _force_lgort,
            _preferABK,
            _auftg.ag_id,
            _vorzug_bedarfsverursacher_gebunden ) ON true
        WHERE
                ( _mit_grundartikel OR artoption_arts__aknr__get <> _auftg.ag_aknr )
            AND tabk.lg_ort__restmenge_kommission__is( lgort, _abix ) IS NOT null
        ORDER BY
          -- #18133 Kommissionierlagerorte von Alternativartikeln kommen vor Standardlagerorten des Geundartikels
          tabk.lg_ort__restmenge_kommission__is( lgort, _abix ) DESC,

          -- #19649 bei Suchoption 'möglichst wenige Chargen' und Bedarfsdeckung möglichst kleine Chargen bevorzugen;
          --        idealerweise passt die Chargengröße genau zum Bedarf
          sort_lagermenge_gleich_bedarf ASC NULLS LAST,

          -- #19649 bei Suchoption 'möglichst wenige Chargen' und Bedarfsdeckung möglichst kleine Chargen bevorzugen;
          --        idealerweise passt die Chargengröße genau zum Bedarf
          sort_lagermenge_groesser_bedarf ASC NULLS LAST,

          -- #19649 bei Suchoption 'möglichst wenige Chargen' und fehlender Bedarfsdeckung möglichst große Chargen bevorzugen
          sort_lagermenge_kleiner_bedarf ASC NULLS LAST,

          -- ansonsten wird die Reihenfolge der Rückgaben von tartikel.art__lag__lg_anztot__by__inputparams_all beibehalten
          ord ASC
      ) LOOP

        -- Abbruchkriterium: Bedarf gedeckt, sofern dieser überhaupt relevant
        IF _bedarf <= 0 AND _mit_bedarf THEN
          exit;
        END IF;

        If _mit_bedarf THEN
            -- Lagerbestand berücksichtigt bereits in der Abfrage reservierte Mengen
            _lganztot_uf1 :=
                 (SELECT me__menge_uf1__in__menge( _auftg.ag_mcv, lg_anztot ) FROM lag WHERE lg_id = _r)
               - coalesce( ( SELECT sum( labk_lagab_uf1 )
                               FROM tlager.lagerabgang__abk__temp
                               WHERE
                                     labk_abix = _abix
                                 AND labk_transaction_timestamp = now()
                                 AND labk_nr_abfrage IS null
                                 AND labk_lgid = _r
                            ), 0
                 );

            -- Kein Bestand? Dann Lagerort übergehen.
            IF coalesce( _lganztot_uf1, 0 ) <= 0 THEN
              continue;
            END IF;

            -- Bedarf verringern
            _bedarf := _bedarf - _lganztot_uf1;

            -- Beim letzten LO wird u.U. nur ein Teil der dort lagernden Menge benötigt, um den Bedarf zu decken
            IF _bedarf < 0 THEN
              _lganztot_uf1 = _lganztot_uf1 + _bedarf;
            END IF;
        END IF;

        -- Rückschreiben in temp-Tabelle
        IF _auftg.ag_id IS NOT null THEN
            INSERT INTO tlager.lagerabgang__abk__temp
              ( labk_abix, labk_agid   , labk_transaction_timestamp, labk_lgid, labk_lagab_uf1              , labk_isAlternativArtikel )
            VALUES
              ( _abix    , _auftg.ag_id, now()                     , _r       , coalesce( _lganztot_uf1, 0 ), _aknr <> _auftg.ag_aknr  );

            _datensatz := true;
        END IF;

        -- #20082 Wenn Suchoption gesetzt, dann nur maximal ein Lagerort pro Artikel.
        IF _ein_lagerort_pro_artikel THEN
            EXIT;
        END IF;

      END LOOP;


      --Pseudo-Lagerort zurückgeben, wenn bislang noch nichts zurückgegeben wurde
      IF NOT _datensatz THEN
            INSERT INTO tlager.lagerabgang__abk__temp
              ( labk_abix, labk_agid   , labk_transaction_timestamp, labk_lgid, labk_lagab_uf1, labk_isAlternativArtikel )
            VALUES
              ( _abix    , _auftg.ag_id, now()                     , 0        , 0             , false                    );
      END IF;

    END $$ LANGUAGE plpgsql;
 --


-- #18078, berechnet die vorgesehen Abgänge eines ABK-bezogenen Lagerabgangs, alle Materiallistenpositionen
CREATE OR REPLACE FUNCTION tabk.auftg__lag__fillup__allpos(
      IN _abix                 integer,
      IN _menge_pro_abk        numeric,  -- wie oft der Artikel für die Fertigung 1 Artikels der ABK benötigt wird
      IN _force_lgort          varchar,
      IN _preferABK            integer,  -- > ag_ownabk obsolet wenn _auftg_ag_id richtig aufgebaut wird
      IN _alle_ag__ag_done     boolean,  -- auch nicht offene Aufträge berücksichtigen? (wenn true alle, sonst nur "NOT ag_done")
      IN _status_teile         varchar,  -- Teilestatus der einbezogenen Aufträge
      IN _chk_halbfertig       boolean,  -- auch Aufträge im Halbfertigstatus einbeziehen?
      IN _suchoption           tabk.lagab_abk_suchoption
    )
    RETURNS VOID AS $$
    DECLARE
      _auftg auftg;
      _matinfo auftgmatinfo;
    BEGIN

      FOR _auftg IN
        SELECT * FROM auftg
        JOIN ( SELECT DISTINCT get_all_child_abk FROM tplanterm.get_all_child_abk( _abix )) AS children ON ag_parentabk = get_all_child_abk
        WHERE
                ( _alle_ag__ag_done OR NOT ag_done )
           AND
               -- Alternativmaterial immer anzeigen, sonst nur den richtigen Status
                (
                     coalesce(ag_post7, '' ) = 'A'
                  OR TSystem.ENUM_GetValue(_status_teile, coalesce(ag_post2, '' )) -- Reihenfolge derzeit wichtig, da Status (Filter aus Oberfläche) mehrere Elemente enthalten kann. Erweiterung das GetValue Listen kann folgt
                  OR nullif(_status_teile, '%') IS null
                )
           AND
                xor( NOT _chk_halbfertig, TSystem.ENUM_GetValue(ag_stat, 'UE' ) )
         ORDER BY ag_pos
      LOOP

        SELECT * FROM auftgmatinfo INTO _matinfo WHERE agmi_ag_id = _auftg.ag_id;
        PERFORM tabk.auftg__lag__fillup(
          _auftg,
          _abix,
          _menge_pro_abk,
          _matinfo,
          _force_lgort,
          _preferABK,
          _suchoption
        );

      END LOOP;

END $$ LANGUAGE plpgsql;
--

-- Liest alle passenden Lagerorte für einen ABK-bezogenen Lagerabgang
--   in Abhängigkeit von Suchoptionen, siehe tabk.auftg__lag__fill__up
CREATE OR REPLACE FUNCTION tabk.auftg__lag__get(
    IN _abix                 integer,  -- ABK-Index
    IN _menge_pro_abk        numeric,  -- wie oft der Artikel für die Fertigung 1 Artikels der ABK benötigt wird
    IN _force_lgort          varchar,  -- erzwungener Lagerort
    IN _preferABK            integer,  -- > ag_ownabk obsolet wenn _auftg_ag_id richtig aufgebaut wird
    IN _alle_ag__ag_done     boolean,  -- auch nicht offene Aufträge berücksichtigen? (wenn true alle, sonst nur "NOT ag_done")
    IN _status_teile         varchar,  -- Teilestatus der einbezogenen Aufträge
    IN _chk_halbfertig       boolean,  -- auch Aufträge im Halbfertigstatus einbeziehen?
    IN _suchoption           tabk.lagab_abk_suchoption = null   -- Suchstrategie, String kodiert, siehe tabk.auftg__lag__suchoptionen__get
                                                                -- null oder '' entspricht der Standardsuche ('ST')
    ) RETURNS TABLE (
        ag_id integer, -- von der Auftragstabelle wird der Kürze wegen nur die ID angegeben
        st_pos integer,
        st_posa varchar,
        stueckl__fertstv_get_konstrstv_ident varchar,
        auftg_ag_aknr varchar,
        ak_nr varchar,
        lang_artmgc_id_iso varchar,
        ag_stk numeric,
        offen numeric,
        term date,
        lg_id integer,
        lg_ort varchar,
        lg_chnr varchar,
        anzlgortaknr integer,
        lg_anztot numeric,
        lg_w_wen integer,
        lg_w_wen_text varchar,
        lagab numeric,
        perf boolean,
        lgortue varchar,
        ak_sernrreq boolean,
        agmi_id integer,
        a2_id integer
    )

    AS $$
    BEGIN

        -- Zwischentabelle füllen
        PERFORM tabk.auftg__lag__fillup__allpos (
            _abix,
            _menge_pro_abk,
            _force_lgort,
            _preferABK,
            _alle_ag__ag_done,
            _status_teile,
            _chk_halbfertig,
            _suchoption
        );

        -- Lagerortsdaten aus Zwischentabelle abrufen
        RETURN QUERY SELECT
               auftg.ag_id,
               stv.st_pos::integer,
               CAST(( chr( 65+stvtrs.ebene ) || '.' || lpad( stvtrs.pos, 5, 0 )) AS VARCHAR(100)) AS st_posa, --archivierte pos (zum Zeitpunkt ABK erstellung)
               tartikel.stueckl__fertstv_get_konstrstv_ident(stv.st_kstv_st_id),
               nullif( auftg.ag_aknr, art.ak_nr )::varchar AS auftg_ag_aknr, -- "Ausgangs/Grundartikelnummer" ist bei Alternativen abweichend. Nur in dem Fall zurückgeben
               art.ak_nr AS ak_nr,
               lang_artmgc_id_iso( auftg.ag_mcv ) AS lang_artmgc_id_iso,
               me__menge_uf1__in__menge( auftg.ag_mcv, auftg.ag_stk_uf1 ) AS ag_stk,
               me__menge_uf1__in__menge( auftg.ag_mcv, auftg.ag_stk_uf1 - auftg.ag_stkl ) AS offen,
               coalesce( auftg.ag_ldatum, auftg.ag_kdatum ) AS term,
               lag.lg_id,
               lag.lg_ort,
               lag.lg_chnr,
               ( SELECT count( DISTINCT lag.lg_ort || lag.lg_chnr )::integer FROM lag WHERE lag.lg_aknr = auftg.ag_aknr AND lag.lg_anztot > 0 ) AS anzlgortaknr,
               me__menge_uf1__in__menge( auftg.ag_mcv, lag.lg_anztot ) AS lg_anztot,

               --Wareneingang
               lag.lg_w_wen,

               --Bestellbezug => Bestellung
               ldsdok__ld_id_ident( lag.lg_ld_id ) AS lg_w_wen_text,
               me__menge_uf1__in__menge( auftg.ag_mcv, numeric_larger( min( labk_lagab_uf1, ( auftg.ag_stk_uf1 - ifthen( _menge_pro_abk IS NULL, auftg.ag_stkl, 0 )) * COALESCE(_menge_pro_abk,1 )), 0 ))
                 AS lagab, -- vorgeschlagene Lagerabgangsmenge
               CASE WHEN TSystem.Settings__GetBool( 'no_neg_lag' ) THEN
                 me__menge_uf1__in__menge( auftg.ag_mcv, auftg.ag_stk_uf1 - auftg.ag_stkl ) <=
                 me__menge_uf1__in__menge( auftg.ag_mcv, numeric_larger(
                     min (lag.lg_anztot, (
                         auftg.ag_stk_uf1 - ifthen( _menge_pro_abk IS null, auftg.ag_stkl, 0 )
                     ) * COALESCE( _menge_pro_abk, 1 )),
                     0 ))
               ELSE
                 ( auftg.ag_stk_uf1 - auftg.ag_stkl ) <= numeric_larger( me__menge_uf1__in__menge( auftg.ag_mcv, ( auftg.ag_stk_uf1 - ifthen (_menge_pro_abk IS null, auftg.ag_stkl, 0 )) * coalesce( _menge_pro_abk, 1 )), 0 )
               END AS perf,
               ( SELECT coalesce( lue_lgort, '' ) FROM lagerortUE   -- Lagerorte UE entspricht inerner PA Nr oder Materiallistennummer (somit Einschränkung auf Ebene?????)
                 WHERE
                      lue_agnr = ( SELECT ld_auftg FROM ldsdok WHERE ld_abk = ab_mainabk )
                   OR lue_agnr = auftg.ag_nr
                 ORDER BY lue_lgort
                 LIMIT 1
               ) AS lgortue,
               art.ak_sernrreq,
               auftgmatinfo.agmi_id,
               ab2.a2_id
          FROM auftg
          JOIN tlager.lagerabgang__abk__temp ON labk_abix = _abix AND labk_transaction_timestamp = now() AND labk_agid = auftg.ag_id AND labk_nr_abfrage IS null
          LEFT JOIN abk ON abk.ab_ix = auftg.ag_ownabk --ab_stat (Sets)
          LEFT JOIN auftgmatinfo ON auftgmatinfo.agmi_ag_id = auftg.ag_id
          LEFT JOIN versart ON v_id = auftgmatinfo.agmi_v_id
          LEFT JOIN stvtrs ON stvtrs.auftg_ag_id = auftg.ag_id
          LEFT JOIN stv ON stv.dbrid = stvtrs.stv_dbrid
               -- findet Lagerorte zu gesuchten Artikeln, ggf. auch zu Alternativarikeln
          LEFT JOIN lag ON lag.lg_id = labk_lgid
               -- Lagerabgang UE: Vorgabecharge nehmen um Lagerzugang zu finden sowie entsprechenden AG
               -- Anzeige von Material mit Arbeitsgangzuordnung
          LEFT JOIN art ON art.ak_nr = lag.lg_aknr
          LEFT JOIN ab2 ON ab2.a2_id IN (
                                      ( SELECT w_a2_id FROM wendat
                                         WHERE
                                               w_aknr = art.ak_nr
                                           AND w_lgort = lag.lg_ort
                                           AND w_lgchnr = auftgmatinfo.agmi_lg_chnr
                                           AND TSystem.Enum_GetValue(ag_stat, 'UE')
                                         ORDER BY lag.lg_w_wen = lag.lg_id DESC
                                         LIMIT 1
                                      ),
                                      auftg.ag_a2_id
                                    )
         WHERE
           auftg.ag_id = labk_agid
         ORDER BY labk_id;

         -- Zwischentabelle aktualisieren
         UPDATE tlager.lagerabgang__abk__temp
         SET labk_nr_abfrage = coalesce(
            ( SELECT max(labk_nr_abfrage) + 1
              FROM tlager.lagerabgang__abk__temp
              WHERE labk_abix = _abix AND labk_transaction_timestamp = now() AND labk_nr_abfrage IS NOT null
            ), 1 )
         WHERE labk_abix = _abix AND labk_transaction_timestamp = now() AND labk_nr_abfrage IS null;

    END $$ LANGUAGE plpgsql;
--


-- Ermitteln aller offenen und erledigten SN zu einem Arbeitsgang
-- auf Basis von:
-- * SN-AG-Mapping  => ( Mapping von lagsernr an ab2 )
-- * Vorgabe-SN     => ( Mapping von lagsernr an ldsdok )
-- * SN-AG-Tracking => ( Mapping von lagsernr an bdea )
CREATE OR REPLACE FUNCTION tabk.mapsernr__ab2__lgs_ids__find(
      IN  _a2_id            integer,
      OUT _lgs_ids_offen    integer[],
      OUT _lgs_ids_erledigt integer[]
    )
  RETURNS record
  AS $$
  DECLARE
    _lgs_ids__ag_spezifisch integer[]; -- SN-AG-Mapping  => ( Mapping von lagsernr an ab2 )
    _lgs_ids__vorgabe       integer[]; -- Vorgabe-SN     => ( Mapping von lagsernr an ldsdok )
    _lgs_ids__getrackt      integer[]; -- SN-AG-Tracking => ( Mapping von lagsernr an bdea )
  BEGIN

    -- alle SN zu seriennummernspezifischem Arbeitsgang holen (SN-AG-Mapping)
    SELECT '{}'::integer[] || array_agg( lagsernr.lgs_id )
      INTO _lgs_ids__ag_spezifisch
      FROM lagsernr
      JOIN mapsernr ON mapsernr.ms_lgs_id = lagsernr.lgs_id
                   AND mapsernr.ms_table = 'ab2'::REGCLASS
     WHERE mapsernr.ms_pkey = _a2_id;

    -- Alle getrackten SN zum aktuellen AG (anhand BDE-Datensatz) holen
    SELECT '{}'::integer[] || array_agg( lagsernr.lgs_id )
      INTO  _lgs_ids__getrackt
      FROM lagsernr
      JOIN mapsernr ON mapsernr.ms_lgs_id = lagsernr.lgs_id
                   AND mapsernr.ms_table = 'bdea'::REGCLASS
      JOIN bdea ON mapsernr.ms_pkey = bdea.ba_id
      JOIN ab2 ON ab2.a2_ab_ix = bdea.ba_ix AND ab2.a2_n = bdea.ba_op
     WHERE ab2.a2_id = _a2_id;

    -- alle Vorgabe-SN zur ABK (anhand BDE-Datensatz) holen
    SELECT '{}'::integer[] || array_agg( lagsernr.lgs_id )
      INTO _lgs_ids__vorgabe
      FROM lagsernr
      JOIN mapsernr ON mapsernr.ms_lgs_id = lagsernr.lgs_id
                   AND mapsernr.ms_table = 'ldsdok'::REGCLASS
      JOIN ldsdok ON ldsdok.ld_id = mapsernr.ms_pkey
      JOIN ab2 ON ab2.a2_ab_ix = ldsdok.ld_abk
     WHERE ab2.a2_id = _a2_id;

    -- offene SN
    _lgs_ids_offen := CASE WHEN cardinality( _lgs_ids__ag_spezifisch ) > 0
                        THEN _lgs_ids__ag_spezifisch - _lgs_ids__getrackt   -- AG ist       SN-spezifisch =>      AG-SN abzüglich getrackte SN entspricht den noch offenen SN
                        ELSE _lgs_ids__vorgabe       - _lgs_ids__getrackt   -- AG ist nicht SN-spezifisch => Vorgabe-SN abzüglich getrackte SN entspricht den noch offenen SN
                        END;
    -- erledigte SN
    _lgs_ids_erledigt := _lgs_ids__getrackt; -- erledigte SN sind alle getrackten SN

  END $$ LANGUAGE plpgsql;


-- Trigger beendet Arbeitsgang wenn alle Vorgabe-Seriennummern zu einem AG per SN-AG-Tracking zurückgemeldet wurden (erledigt oder Ausschuss) #19110
-- Prodat-Wiki: Lager-Seriennummern
CREATE OR REPLACE FUNCTION mapsernr__a_iu__bdea()  RETURNS trigger
    AS $$
  DECLARE
    _a2_id                 integer;
  BEGIN

    -- Arbeitsgang zu aktuellem BDE-Datensatz holen
    SELECT a2_id
      INTO _a2_id
      FROM bdea
      JOIN ab2 ON ab2.a2_ab_ix = bdea.ba_ix
              AND ab2.a2_n = bdea.ba_op
     WHERE new.ms_pkey = bdea.ba_id
       AND new.ms_table = 'bdea'::REGCLASS;

    -- Wenn es erledigte SN zum Arbeitsgang gibt und keine mehr offen sind => dann Arbeitsgang abschließen
    IF ( SELECT cardinality( _lgs_ids_offen ) = 0
            AND cardinality( _lgs_ids_erledigt ) > 0
           FROM tabk.mapsernr__ab2__lgs_ids__find( _a2_id ) as seriennummern )
    THEN
      UPDATE ab2 SET a2_ende = TRUE
       WHERE a2_id = _a2_id
         AND NOT a2_ende;
    END IF;

    RETURN new;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER mapsernr__a_iu__bdea
    AFTER INSERT OR UPDATE OF ms_table
    ON mapsernr
    FOR EACH ROW
    WHEN (new.ms_table::oid = 'bdea'::regclass::oid)
    EXECUTE PROCEDURE mapsernr__a_iu__bdea();
--

-- Funktion die zu einem gegebenen Artikel den letzten berechneten Durchschnittspreis aus dem Lagerlog findet
--
CREATE OR REPLACE FUNCTION TLager.lagerlog__lo_adpreis__get_last_before_timestamp(
    IN _ak_nr        varchar,
    IN _lo_datetime  timestamp DEFAULT current_timestamp
    )
    RETURNS          numeric
  AS $$
  DECLARE
    last_lo_adpreis  numeric(20, 4);
  BEGIN

    -- Wir suchen den letzten berechneten Durchschnittspreis zu diesem Lagerlog-Satz (Umsetzung analog D-Wert aus Lagerbewertung -> FUNCTION lagerlog_stichtagbestand).
    SELECT lo_adpreis
      INTO last_lo_adpreis
      FROM lagerlog
     WHERE lo_aknr_new = _ak_nr
       AND lo_datum    < _lo_datetime
       AND lo_adpreis  IS NOT null
     ORDER BY lo_id DESC
     LIMIT 1;

    RETURN last_lo_adpreis;

  END $$ LANGUAGE plpgsql STABLE;
--
